# -*- coding: utf-8 -*-

import re


# Список регулярных выражений для номеров телефонов, которые нужно проверить только на возможное существование
# (флаг is_possible), но не нужно проверять на полную валидность (флаг is_valid). Требуется для возможности работать
# с номерами телефонов пользователей, которые уже были когда-то добавлены в базу - то есть они были каким-либо образом
# подтверждены (возможно, вследствие бага или с помощью брутфорса кода подтверждения) и успешно привязаны к аккаунту.
# Без whitelist'а при создании объекта телефона с таким номером будет выкидываться исключение даже при обычном чтении
# номеров из базы с помощью ЧЯ. Также, нужно дать возможность пользователям подтверждать номера, которые не считаются
# валидными в libphonenumber, но на самом деле существуют - обычно мы про такое узнаём по прямой жалобе в саппорт.
#
# Список состоит из правил-массивов, элементами которого являются:
#   1. Полноценное регулярное выражение, под которое подпадает либо целый класс номеров, либо один конкретный номер.
#      Желательно разбивать выражение на части с помощью пробелов (они не будут учитываться):
#      код страны, префикс, остальная часть номера.
#   2. Короткое, человеко-читаемое название этого правила (может попасть в логи).
# Сортировка в списке ведётся по коду страны в номере.
# У каждого правила должен быть пример настоящего полного номера.
# Также, нужно не забыть добавить пример номера в авто-тесты (см. test_parse_invalid_whitelisted_ok
# в core/tests/types/phone_number.py).
#
# Перед каждым правилом желательно написать о причинах его добавления с имеющимися на руках подробностями, например:
#   * нормативные документы министерства связи страны о выделении номерной ёмкости
#     или о миграции со старого плана нумерации на новый;
#   * новость о том же самом;
#   * страница конкретного оператора;
#   * тикет в OTRS с жалобой от настоящего владельца существующего номера;
#   * сайты, содержащие подобный контактный номер;
#   * история отправленных смс в паспортной админке;
#   * любые другие ссылки - сторонние сайты с базой номерных ёмкостей, страницы в Википедии.
#
# Чем меньшее количество номеров захватывает правило - тем лучше. Нужно помнить, что весь этот whitelist - это, по сути,
# костыль, т.к. libphonenumber в идеале должен быть всегда актуальным, но это не так. То есть, желатально стараться
# составлять регулярное выражение как можно конкретнее. Но также не следует просто добавлять каждый номер
# по-отдельности, если при этом есть понимание, что добавлять нужно номера с определённым префиксом - тогда другие
# похожие номера не будут заблокированы в будущем.
#
# При наличии весомых доказательств о новом правиле нумерации, желательно сообщить об этом в апстрим libphonenumber:
# либо создав issue, либо сделать полноценный pull request в https://github.com/googlei18n/libphonenumber. Если правило
# будет добавлено в основную базу, нужно дождаться апдейта python-биндинга, обновиться, убедиться, что номера стали
# валидными - после чего удалить правило из списка.
PHONE_NUMBERS_WHITELIST = [
    # Очень старые номера Узбекистана, самая свежая дата подтверждения у которых аж 2012-07-10.
    # Пример: +998613562752
    [r'^ \+998 61 35 \d{5} $', 'uzbekistan_6135'],

    # PASSP-16826
    # TODO: Нужно преобразовать, смотри тикет.
    [r'^ \+382 67 9 \d{6} $', 'montenegro_382679'],

    [r'^ \+55 [1-9] \d{9} $', 'brazil2016_old_mobiles'],
    [r'^ \+55 [1-9] \d{10} $', 'brazil2016_new_mobiles'],

    [r'^ \+972 5 7 \d{7} $', 'israel_97257'],

    # Старые номера Вьетнама
    [r'^ \+84 \d{10} $', 'vietnam_old_11_digits'],

    # Новые номера Кот-Д'Ивуара
    [r'^ \+225 (?: 0[157] | 2[157] ) \d{8} $', 'ivory_coast_10_digits'],
    # Старые номера Кот-Д'Ивуара
    [r'^ \+225 \d{8} $', 'ivory_coast_8_digits'],
    # Платные номера Белоруси + неверно парсятся libphonenumber
    [r'^ \+375 810 \d{7} $', 'belarus_premium_rate'],
]
for rule in PHONE_NUMBERS_WHITELIST:
    rule[0] = re.compile(rule[0], re.VERBOSE)

# Отключение правила ведёт к тому, что
# * Номер не попадёт в белый список
# * Если libphonenumber считает номер действительным, то это будет
#   проигнорировано и такой номер будет считаться недействительным.
PHONE_NUMBERS_WHITELIST_DISABLED_RULES = {
    'belarus_premium_rate',
}
