# -*- coding: utf-8 -*-
import base64
import hashlib
from math import ceil
from random import randint
import string
import time

from passport.backend.core.cookies.lrandoms import get_lrandoms_manager
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
import six
from six.moves import zip_longest


class CookieLBaseError(BaseCoreError):
    pass


class CookieLPackError(CookieLBaseError):
    pass


class CookieLUnpackError(CookieLBaseError):
    pass


class CookieL(object):
    VERSION = 3

    COOKIE_L_FIELDS_NUMBER = 5
    COOKIE_L_HEADER_LENGTH = 4

    COOKIE_L_LOGIN_PART_OFFSET = 24
    COOKIE_L_DATA_LENGTH = 64

    def __init__(self, lrandoms_manager=None):
        self.lrandoms_manager = lrandoms_manager or get_lrandoms_manager()

    def _ord(self, c):
        if isinstance(c, int):
            return c
        return ord(c)

    def _chr(self, c):
        if isinstance(c, int):
            return chr(c)
        return c

    def string_xor(self, data, key, current_version):
        if current_version == self.VERSION:
            data_length = len(data)
            key_length = len(key)
            multiplier = int(ceil(float(data_length) / key_length))
            key = (key * multiplier)[:data_length]

        byte_codes = [
            self._ord(x) ^ self._ord(y)
            for x, y in zip_longest(data, key, fillvalue='0')
        ]
        if six.PY2:
            return ''.join(chr(c) for c in byte_codes)
        else:
            return bytes(byte_codes)

    def pack(self, uid, login, lrandom=None):
        uid_str = str(uid)
        login_str = login
        if six.PY2:
            login_str = login_str.encode('utf-8')

        lrandom = lrandom or self.lrandoms_manager.current_lrandom()
        if lrandom is None:
            raise CookieLPackError('LRandom not loaded')

        lkey = lrandom['body']
        lkey_id = lrandom['id']

        uid_length = len(uid_str)

        linfo = ''

        space = string.digits + string.ascii_letters
        for i in [0, uid_length, 0, 0]:
            linfo += space[i]

        linfo += uid_str
        while len(linfo) < self.COOKIE_L_LOGIN_PART_OFFSET:
            linfo += str(randint(0, 9))

        linfo += login_str

        current_version = self.VERSION
        linfo = self.string_xor(smart_bytes(linfo), lkey, current_version)

        linfo = base64.b64encode(smart_bytes(linfo)).decode('utf8')

        fields = [linfo, str(int(time.time())), str(lkey_id), str(current_version) + str(randint(0, 100000 - 1))]
        linfo = '.'.join(fields)

        return '%s.%s' % (linfo, hashlib.md5(smart_bytes(linfo + lkey)).hexdigest())

    def unpack(self, cookie_l):
        cookie_l = smart_text(cookie_l)

        base62_chars = string.digits + string.ascii_letters
        base62_codes = dict((base62_chars[i], i) for i, _ in enumerate(base62_chars))

        fields = cookie_l.split('.')

        if len(fields) != self.COOKIE_L_FIELDS_NUMBER:
            raise CookieLUnpackError('Waiting for %s fields but got %s' % (
                self.COOKIE_L_FIELDS_NUMBER,
                len(fields),
            ))

        encoded_data, timestamp, lkey_id, random, signature = fields

        version = random[:1]
        try:
            version = int(version)
        except ValueError:
            raise CookieLUnpackError("Bad version: '%s'" % version)

        encoded_data = encoded_data.replace(' ', '+')

        lrandom = self.lrandoms_manager.get_lrandom(lkey_id)

        if not lrandom:
            raise CookieLUnpackError('Unknown lrandom lkey_id: %s' % lkey_id)

        signature_base_string = '.'.join([encoded_data, timestamp, lkey_id, random]) + lrandom['body']
        real_signature = hashlib.md5(signature_base_string.encode('utf8')).hexdigest()

        if real_signature != signature:
            raise CookieLUnpackError("Signature mismatch: '%s' should be '%s'" % (real_signature, signature))

        decoded_data = base64.b64decode(encoded_data)
        data = self.string_xor(decoded_data, lrandom['body'], version)

        header = data[0:self.COOKIE_L_HEADER_LENGTH]
        uid_junk_length, uid_length, login_junk_length, login_length = [base62_codes[self._chr(i)] for i in header]

        uid_offset = self.COOKIE_L_HEADER_LENGTH + uid_junk_length
        login_offset = self.COOKIE_L_LOGIN_PART_OFFSET + login_junk_length

        uid_str = data[uid_offset:uid_offset + uid_length]
        if version < self.VERSION:
            login_str = data[login_offset:login_offset + login_length]
        else:
            login_str = data[login_offset:]

        try:
            uid = int(uid_str)
        except ValueError:
            raise CookieLUnpackError('Bad uid value: %s' % uid_str)

        try:
            login = login_str.decode('utf-8')
        except UnicodeDecodeError:
            raise CookieLUnpackError('Bad login value: %s' % login_str)

        return {
            'uid': uid,
            'login': login,
        }
