# -*- coding: utf-8 -*-

import hashlib
from itertools import cycle
import re
import time

from passport.backend.core.builders.blackbox import (
    BLACKBOX_CHECK_SIGN_STATUS_OK,
    get_blackbox,
)
from passport.backend.core.conf import settings
from passport.backend.core.cookies.lrandoms import get_lrandoms_manager
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import (
    from_base64_url,
    to_base64_url,
)
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
import six
from six.moves import zip


VERSION_GTE2 = re.compile(r'^(\d+):')


def _ord(c):
    if isinstance(c, int):
        return c
    return ord(c)


def string_xor(data, key):
    byte_codes = [
        _ord(x) ^ _ord(y)
        for x, y in zip(data, cycle(key))
    ]
    if six.PY2:
        return ''.join(chr(c) for c in byte_codes)
    else:
        return bytes(byte_codes)


class BaseSignedEncryptedCookie(object):
    V1_COOKIE_FIELDS_NUMBER = 5

    pack_error = None
    unpack_error = None

    def pack(
        self,
        data,
        lrandom=None,
    ):
        if settings.LAH_SIGN_VERSION == 2:
            return self.pack_v2(data)
        if settings.LAH_SIGN_VERSION == 1:
            return self.pack_v1(data, lrandom)
        raise NotImplementedError("Don't know how to pack version: %s" % settings.LAH_SIGN_VERSION)

    def unpack(self, cookie):
        cookie = smart_text(cookie)

        version = self.get_version(cookie)
        if version == 2:
            return self.unpack_v2(cookie)
        if version == 1:
            return self.unpack_v1(cookie)

        raise self.unpack_error('Unknown version: "%s"' % version)

    def get_version(self, cookie_bytes):
        version_gte2 = VERSION_GTE2.match(cookie_bytes)
        if version_gte2:
            return int(version_gte2.group(1))

        if self.is_version1(cookie_bytes):
            return 1

        raise self.unpack_error('Failed to get version: ' + smart_text(cookie_bytes))

    def pack_v2(self, data):
        data = smart_bytes(data)
        encoded_data = to_base64_url(data)
        response = get_blackbox().sign(
            value=encoded_data,
            ttl=int(settings.LAH_SIGN_TTL.total_seconds()),
            sign_space=settings.LAH_SIGN_SPACE,
        )
        return '2:' + response.get('signed_value')

    def unpack_v2(self, cookie):
        _, signed_data = cookie.split(':', 1)
        response = get_blackbox().check_sign(
            sign_space=settings.LAH_SIGN_SPACE,
            signed_value=signed_data,
        )
        if response.get('status') != BLACKBOX_CHECK_SIGN_STATUS_OK:
            raise self.unpack_error('Lah signature is invalid: %s' % response.get('status'))

        encoded_data = response.get('value')
        return from_base64_url(smart_bytes(encoded_data))

    @cached_property
    def lrandoms_manager(self):
        return get_lrandoms_manager()

    def pack_v1(self, data, lrandom=None):
        data = smart_bytes(data)

        lrandom = lrandom or self.lrandoms_manager.current_lrandom()
        if lrandom is None:
            raise self.pack_error('LRandom not loaded')

        lkey = lrandom['body']
        lkey_id = lrandom['id']

        encoded_data = to_base64_url(string_xor(data, lkey)).decode('utf8')
        version = 1

        fields = [encoded_data, int(time.time()), lkey_id, version]
        body = '.'.join(map(str, fields))
        digest = hashlib.md5(smart_bytes(body + lkey)).hexdigest()

        return '%s.%s' % (body, digest)

    def unpack_v1(self, cookie):
        fields = cookie.split('.')

        if len(fields) != self.V1_COOKIE_FIELDS_NUMBER:
            raise self.unpack_error('Waiting for %s fields but got %s' % (
                self.V1_COOKIE_FIELDS_NUMBER,
                len(fields),
            ))

        encoded_data, timestamp, lkey_id, _, signature = fields
        version = 1

        lrandom = self.lrandoms_manager.get_lrandom(lkey_id)

        if not lrandom:
            raise self.unpack_error('Unknown lrandom lkey_id: %s' % lkey_id)

        signature_base_string = '.'.join([encoded_data, timestamp, lkey_id, str(version)]) + lrandom['body']
        real_signature = hashlib.md5(signature_base_string.encode('utf8')).hexdigest()

        if real_signature != signature:
            raise self.unpack_error('Signature mismatch: "%s" should be "%s"' % (real_signature, signature))

        decoded_data = from_base64_url(encoded_data.encode('utf8'))
        data = string_xor(decoded_data, lrandom['body'])

        return data

    def is_version1(self, cookie):
        fields = cookie.split('.')

        try:
            version = fields[3]
        except IndexError:
            return False

        try:
            version = int(version)
        except ValueError:
            return False

        return version == 1
