# -*- coding: utf-8 -*-
from collections import namedtuple

from cookiemy import Cookiemy
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.utils.string import smart_str
from six import iteritems


COOKIEMY_BLOCKS = {
    39: 'language',
    54: 'long_session',
}

COOKIEMY_CODE_TO_LANGUAGE = {
    1: 'ru',
    2: 'uk',
    3: 'en',
    4: 'kk',
    5: 'be',
    6: 'tt',
    7: 'az',
    8: 'tr',
    9: 'hy',
    10: 'ka',
    11: 'ro',
}

COOKIEMY_LANGUAGE_TO_CODE = dict((code, key) for key, code in iteritems(COOKIEMY_CODE_TO_LANGUAGE))


def build_cookiemy_blocks():
    names = [name for id, name in sorted(iteritems(COOKIEMY_BLOCKS))]
    keys = sorted(COOKIEMY_BLOCKS.keys())
    return namedtuple('CookieMyBlock', names)._make(keys)


CookieMyBlock = build_cookiemy_blocks()


class CookieMyBaseError(BaseCoreError):
    pass


class CookieMyPackError(CookieMyBaseError):
    """Не удается собрать параметры в последовательность"""


class CookieMyUnpackError(CookieMyBaseError):
    """Не получилось разобрать тело куки"""


class CookieMyValueError(CookieMyBaseError):
    """Попытка записи недопустимого значения"""


class CookieMyWrapper(object):
    """Обертка для работы с cookie `my`"""

    def __init__(self):
        self.cookie = Cookiemy()

    def keys(self):
        return self.cookie.keys()

    def validate_id(self, block_id):
        if block_id not in range(0, 255):
            raise CookieMyValueError('Block ID=%d not in [0, 255]' % block_id)

    def validate_value(self, value_list):
        if any(x < 0 for x in value_list):
            raise CookieMyValueError('Value contains negative numbers')

    def __getattr__(self, item):
        block_id = getattr(CookieMyBlock, item)
        return self.cookie.find(block_id)

    def __setattr__(self, key, value):
        if key in CookieMyBlock._fields:
            block_id = getattr(CookieMyBlock, key)
            self.validate_id(block_id)
            self.validate_value(value)
            self.cookie.insert(block_id, value)
        else:
            super(CookieMyWrapper, self).__setattr__(key, value)

    def pack(self):
        """Собирает последовательность битов в строчку"""
        try:
            return self.cookie_to_string()
        except RuntimeError as ex:
            raise CookieMyPackError(ex)

    def cookie_to_string(self):
        return str(self.cookie)

    def unpack(self, cookie_value):
        """Парсит переданное значение для дальнейшей работы"""
        try:
            self.cookie.parse(smart_str(cookie_value))
        except RuntimeError as ex:
            raise CookieMyUnpackError(ex)


def _bool_convert(value):
    """Удобно записывает булево значение в блок куки my"""
    return [int(value)]


def _parse_to_bool(value):
    """Удобно прочитает булево значение флажка из блока куки my"""
    if len(value) == 0:
        return None
    return bool(value[0])


class CookieMy(object):

    def __init__(self, cookie_value=''):
        self._cookie = CookieMyWrapper()
        self._cookie.unpack(cookie_value)

    @property
    def language(self):
        """Блок №39 в куке. См. описание на вики
        https://wiki.yandex-team.ru/MyCookie/%D0%9D%D0%BE%D0%BC%D0%B5%D1%80%D0%91%D0%BB%D0%BE%D0%BA%D0%B0
        """
        values = self._cookie.language
        if len(values) < 2:
            return None

        # "По историческим причинам" используется второй элемент из массива значений в этом блоке - см.вики
        actual_value = values[1]
        return COOKIEMY_CODE_TO_LANGUAGE.get(actual_value)

    @language.setter
    def language(self, new_value):
        actual_value = COOKIEMY_LANGUAGE_TO_CODE.get(new_value, 0)
        # TODO: Узнать, нужно ли сохранять значение первого элемента в этом блоке
        values = [0, actual_value]
        self._cookie.language = values

    @property
    def long_session(self):
        """Блок №54 в куке `my`"""
        value = self._cookie.long_session
        return _parse_to_bool(value)

    @long_session.setter
    def long_session(self, new_value):
        self._cookie.long_session = _bool_convert(new_value)

    def pack(self):
        return self._cookie.pack()


__all__ = (
    'CookieMyBaseError',
    'CookieMyPackError',
    'CookieMyUnpackError',
    'CookieMyValueError',
    'CookieMyWrapper',
    'CookieMy',
)
