# -*- coding: utf-8 -*-
from collections import OrderedDict

from passport.backend.core.exceptions import BaseCoreError
from passport.backend.utils.string import (
    always_str,
    smart_bytes,
    smart_text,
)
import six
from six.moves.urllib.parse import (
    quote,
    unquote,
)


class CookieYBaseError(BaseCoreError):
    pass


class CookieYUnpackError(CookieYBaseError):
    pass


class CookieYPackError(CookieYBaseError):
    pass


class CookieY(object):
    """
    Y-кука - это кука-контейнер. Есть два типа y-куки - сессионная и перманентная

    Описание формата: https://wiki.yandex-team.ru/Cookies/Y
    """
    def __init__(self,
                 fields_order,
                 fields_parsers=None,
                 fields_packers=None,
                 unique_field=None,
                 blocks_delimiter='#',
                 fields_delimiter='.'):
        self.fields_order = fields_order
        self.fields_parsers = fields_parsers or {}
        self.fields_packers = fields_packers or {}
        self.unique_field = unique_field
        self.blocks_delimiter = blocks_delimiter
        self.fields_delimiter = fields_delimiter

    @staticmethod
    def unpack_urlencoded_value(value):
        try:
            return smart_text(unquote(value))
        except UnicodeDecodeError:
            raise CookieYUnpackError(u'Bad urlencode value: %s' % value)

    @staticmethod
    def pack_urldecoded_value(value):
        return quote(smart_bytes(value), safe='/:')

    @staticmethod
    def unpack_int_value(value):
        try:
            return int(value)
        except ValueError:
            raise CookieYUnpackError(u'Not integer: %s' % smart_text(value))

    @staticmethod
    def pack_int_value(value):
        if not isinstance(value, six.integer_types):
            raise CookieYPackError(u'Not integer value: %s' % smart_text(value))
        return str(value)

    def __unpack_block(self, block):
        block_values = block.split(self.fields_delimiter, len(self.fields_order) - 1)
        if len(block_values) != len(self.fields_order):
            raise CookieYUnpackError(u'Waiting for %s fields but got %s: %s' % (
                len(self.fields_order),
                len(block_values),
                smart_text(block),
            ))

        result = {}
        for field_name, value in zip(self.fields_order, block_values):
            parser = self.fields_parsers.get(field_name, lambda x: x)
            parsed_value = parser(value)
            result[field_name] = parsed_value
        return result

    def __pack_block(self, unpacked_block):
        if sorted(unpacked_block.keys()) != sorted(self.fields_order):
            raise CookieYPackError(u'Waiting for %s fields but got %s in block %s' % (
                smart_text(self.fields_order),
                smart_text(unpacked_block.keys()),
                smart_text(unpacked_block),
            ))
        packed_fields = []
        for field_name in self.fields_order:
            packer = self.fields_packers.get(field_name, lambda x: x)
            value = unpacked_block[field_name]
            packed_value = packer(value)
            packed_fields.append(packed_value)
        return self.fields_delimiter.join(packed_fields)

    def unpack(self, cookie_y):
        cookie_y = smart_text(cookie_y) if six.PY3 else smart_bytes(cookie_y)
        if not cookie_y:
            return [], ''

        blocks = cookie_y.split(self.blocks_delimiter)

        unpacked_blocks = []
        unpacked_count = 0
        for block in blocks:
            try:
                unpacked_block = self.__unpack_block(block)
                unpacked_count += 1
                unpacked_blocks.append(unpacked_block)
            except CookieYUnpackError:
                break
        not_unpacked_tail = self.blocks_delimiter.join(blocks[unpacked_count:])
        return unpacked_blocks, not_unpacked_tail

    def pack(self, blocks):
        packed_blocks = []
        unique_fields = set()
        for block in blocks:
            if self.unique_field:
                unique_field_value = block[self.unique_field]
                if unique_field_value in unique_fields:
                    raise CookieYPackError(u'Value of the field `%s` must be unique: %s' % (self.unique_field, unique_field_value))
                unique_fields.add(unique_field_value)
            packed_block = self.__pack_block(block)
            packed_blocks.append(packed_block)
        return self.blocks_delimiter.join(packed_blocks)


class NameAndValueCookie(CookieY):
    FIELDS_ORDER = [
        'name',
        'value',
    ]

    FIELDS_PARSERS = {
        'name': CookieY.unpack_urlencoded_value,
        'value': CookieY.unpack_urlencoded_value,
    }

    FIELDS_PACKERS = {
        'name': CookieY.pack_urldecoded_value,
        'value': CookieY.pack_urldecoded_value,
    }

    UNIQUE_FIELD = 'name'

    def __init__(self):
        super(NameAndValueCookie, self).__init__(
            self.FIELDS_ORDER,
            self.FIELDS_PARSERS,
            self.FIELDS_PACKERS,
            self.UNIQUE_FIELD,
        )


class TestCookie(NameAndValueCookie):
    """Кука для тестирования функционала Паспорта"""


class SessionCookieY(NameAndValueCookie):
    """Кука для хранения различных параметров на время сессии пользователя"""


class PermanentCookieY(CookieY):
    FIELDS_ORDER = [
        'expire',
        'name',
        'value',
    ]

    FIELDS_PARSERS = {
        'expire': CookieY.unpack_int_value,
        'name': CookieY.unpack_urlencoded_value,
        'value': CookieY.unpack_urlencoded_value,
    }

    FIELDS_PACKERS = {
        'expire': CookieY.pack_int_value,
        'name': CookieY.pack_urldecoded_value,
        'value': CookieY.pack_urldecoded_value,
    }

    UNIQUE_FIELD = 'name'

    def __init__(self):
        super(PermanentCookieY, self).__init__(
            self.FIELDS_ORDER,
            self.FIELDS_PARSERS,
            self.FIELDS_PACKERS,
            self.UNIQUE_FIELD,
        )


class CookieYContainer(object):
    def __init__(self, parser):
        if not parser.unique_field:
            raise ValueError('Need parser with unique field')
        self._blocks = None
        self._parser = parser
        self.reset()

    def reset(self):
        self._blocks = OrderedDict()

    def parse(self, cookie_y):
        self.reset()
        blocks, _not_unpacked_tail = self._parser.unpack(cookie_y)
        for block in blocks:
            unique_field_value = block[self._parser.unique_field]
            self._blocks[unique_field_value] = block
        return self

    def serialize(self):
        return self._parser.pack([self._blocks[key] for key in self._blocks.keys()])

    def insert(self, block):
        unique_field_value = block[self._parser.unique_field]
        self._blocks[unique_field_value] = dict(block)
        return True

    def get(self, unique_field_value):
        return dict(self._blocks.get(unique_field_value, {}))

    def erase(self, unique_field_value):
        return self._blocks.pop(unique_field_value, None)

    def __iter__(self):
        return six.iteritems(self._blocks)


class NameAndValueCookieContainer(CookieYContainer):
    """Объект для работы с куками ключ-значение"""
    parser = NameAndValueCookie()

    def __init__(self):
        super(NameAndValueCookieContainer, self).__init__(self.parser)

    def insert(self, name, value):
        return super(NameAndValueCookieContainer, self).insert(
            {'name': name, 'value': value},
        )


class SessionCookieYContainer(NameAndValueCookieContainer):
    parser = SessionCookieY()


class TestCookieContainer(NameAndValueCookieContainer):
    parser = TestCookie()


class PermanentCookieYContainer(CookieYContainer):
    def __init__(self):
        super(PermanentCookieYContainer, self).__init__(PermanentCookieY())

    def insert(self, name, value, expire):
        return super(PermanentCookieYContainer, self).insert(
            {'name': name, 'value': value, 'expire': expire},
        )


class YpSpSubvalueParser(object):
    # У этого отдела куки разделители - между ключем и значением - двоеточие, между парами - двоеточие.
    # Потому делаем свой парсер
    unique_field = 'name'

    def unpack(self, cookie_value):
        cookie_value = always_str(cookie_value)
        if not cookie_value:
            return [], ''
        result = []
        not_unpacked_tail = ''
        splited_values = cookie_value.split(':')
        if len(splited_values) % 2 != 0:
            not_unpacked_tail = splited_values.pop()
        for i in range(0, len(splited_values), 2):
            result.append({self.unique_field: splited_values[i], 'value': splited_values[i + 1]})
        return result, not_unpacked_tail

    def pack(self, blocks):
        # соединить по двоеточию
        result = []
        for block in blocks:
            result.append(':'.join([block[self.unique_field], block['value']]))
        return ':'.join(result)


class YpSpSubvalueContainer(NameAndValueCookieContainer):
    parser = YpSpSubvalueParser()
