# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    Blackbox,
    BlackboxInvalidResponseError,
)
from passport.backend.core.conf import settings
from passport.backend.core.disk_cache import (
    DiskCache,
    DiskCacheReadError,
    DiskCacheWriteError,
)
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)


log = logging.getLogger('passport.cookies.lrandoms')


@lazy_loadable('LRandomsManager')
class _LRandomsManager(object):
    def __init__(self, blackbox=None, cache_time=None, expire_offset=None):
        self._blackbox = blackbox or Blackbox()

        self._expired_time = 0
        self._timesorted_lrandoms = []
        self._key_ids = {}
        self._cache_time = cache_time or settings.LRANDOMS_CACHE_TIME
        self._expire_offset = expire_offset or settings.LRANDOMS_EXPIRE_OFFSET

    @property
    def need_reload(self):
        return bool(not self._timesorted_lrandoms or time.time() > self._expired_time)

    def load(self):
        '''Возвращает True, если данные были прочитаны из ЧЯ, False иначе'''
        if not self.need_reload:
            return False

        now = time.time()
        lrandoms_reloaded = False
        lrandoms = None
        try:
            lrandoms = self._blackbox.lrandoms()
            if not lrandoms:
                raise BlackboxInvalidResponseError("Empty lrandoms list")
            lrandoms_reloaded = True
            DiskCache('lrandoms').dump(lrandoms)
        except BaseBlackboxError as e:
            log.warning('Could not load lrandoms from blackbox: %s', e)
            if not self._timesorted_lrandoms:
                try:
                    lrandoms = DiskCache('lrandoms').load()
                except DiskCacheReadError as e:
                    log.warning('Could not load lrandoms from file: %s', e)
        except DiskCacheWriteError as e:
            log.warning('Could not dump lrandoms to file: %s', e)

        if lrandoms:
            self._timesorted_lrandoms = sorted(lrandoms, key=lambda x: x['created_timestamp'], reverse=True)
        if lrandoms_reloaded:
            self._key_ids = dict((lrandom['id'], lrandom) for lrandom in lrandoms)
            # Время протухания ключей округляется в меньшую сторону по
            # границе кратной _cache_time + дополнительный запас по времени
            self._expired_time = (time.time() + self._cache_time -
                                  now % self._cache_time + self._expire_offset)
            log.info('Using new lrandoms')
        else:
            log.info('Using old lrandoms')
        return lrandoms_reloaded

    def current_lrandom(self):
        # Ключи приходят по два (два генерящих сервера) за последние три года
        # Берем не самый свежий с учетом того, чтобы lrandoms.txt успел доставиться на все машины
        # и что количество генерящих их серверов может увеличиться
        if not self._timesorted_lrandoms:
            return None
        idx = min(5, len(self._timesorted_lrandoms) - 1)
        return self._timesorted_lrandoms[idx]

    def get_lrandom(self, key_id):
        return self._key_ids.get(key_id)

    def __str__(self):
        return '_LRandomsManager: expired_time=%s' % self._expired_time


def get_lrandoms_manager():
    manager = LazyLoader.get_instance('LRandomsManager')
    manager.load()
    return manager
