# -*- coding: utf-8 -*-
from datetime import timedelta
from time import time
import warnings

from passport.backend.core.cookies.utils.werkzeug_ports._compat import (
    iter_bytes,
    text_type,
    to_bytes,
    to_unicode,
    try_coerce_native,
)
from passport.backend.core.cookies.utils.werkzeug_ports._internals import (
    _cookie_parse_impl,
    _cookie_quoting_map,
    _legal_cookie_chars,
    _make_cookie_domain,
)
from passport.backend.core.cookies.utils.werkzeug_ports.urls import iri_to_uri
import six
from werkzeug.http import http_date


MAX_COOKIE_EXPIRATION_TIMESTAMP = 2 ** 31 - 1


def _cookie_quote(b):
    buf = bytearray()
    all_legal = True
    _lookup = _cookie_quoting_map.get
    _push = buf.extend

    for char in iter_bytes(b):
        if char not in _legal_cookie_chars:
            all_legal = False
            char = _lookup(char, char)
        _push(char)

    if all_legal:
        return bytes(buf)
    return bytes(b'"' + buf + b'"')


# copy-paste from werkzeug.http.dump_cookie. Отличия:
# 1) значение куки не экранируется
# 2) expires форматируется с помощью http_date, а не cookie_date
# 3) expires проверяется не на None, а на bool
# 4) дополнительно кодирую value, если оно типа unicode
def dump_cookie(key, value='', max_age=None, expires=None, path='/',
                domain=None, secure=False, httponly=False,
                charset='utf-8', sync_expires=True, max_size=4093,
                samesite=None):
    """Creates a new Set-Cookie header without the ``Set-Cookie`` prefix
    The parameters are the same as in the cookie Morsel object in the
    Python standard library but it accepts unicode data, too.

    On Python 3 the return value of this function will be a unicode
    string, on Python 2 it will be a native string.  In both cases the
    return value is usually restricted to ascii as the vast majority of
    values are properly escaped, but that is no guarantee.  If a unicode
    string is returned it's tunneled through latin1 as required by
    PEP 3333.

    The return value is not ASCII safe if the key contains unicode
    characters.  This is technically against the specification but
    happens in the wild.  It's strongly recommended to not use
    non-ASCII values for the keys.

    :param max_age: should be a number of seconds, or `None` (default) if
                    the cookie should last only as long as the client's
                    browser session.  Additionally `timedelta` objects
                    are accepted, too.
    :param expires: should be a `datetime` object or unix timestamp.
    :param path: limits the cookie to a given path, per default it will
                 span the whole domain.
    :param domain: Use this if you want to set a cross-domain cookie. For
                   example, ``domain=".example.com"`` will set a cookie
                   that is readable by the domain ``www.example.com``,
                   ``foo.example.com`` etc. Otherwise, a cookie will only
                   be readable by the domain that set it.
    :param secure: The cookie will only be available via HTTPS
    :param httponly: disallow JavaScript to access the cookie.  This is an
                     extension to the cookie standard and probably not
                     supported by all browsers.
    :param charset: the encoding for unicode values.
    :param sync_expires: automatically set expires if max_age is defined
                         but expires not.
    :param max_size: Warn if the final header value exceeds this size. The
        default, 4093, should be safely `supported by most browsers
        <cookie_>`_. Set to 0 to disable this check.
    :param samesite: Limits the scope of the cookie such that it will only
                     be attached to requests if those requests are "same-site".

    .. _`cookie`: http://browsercookielimits.squawky.net/
    """

    key = to_bytes(key, charset)
    value = to_bytes(value, charset)

    # этой переменной в коде werkzeug не было
    add_expires = False

    if path is not None:
        path = iri_to_uri(path, charset)
    domain = _make_cookie_domain(domain)
    if isinstance(max_age, timedelta):
        max_age = (max_age.days * 60 * 60 * 24) + max_age.seconds
    if expires:
        if not isinstance(expires, (str,)):
            expires = http_date(expires)
        add_expires = True
    elif max_age is not None and sync_expires:
        expires = to_bytes(http_date(time() + max_age))
        add_expires = True

    samesite = samesite.title() if samesite else None
    # Здесь 'None' - явная директива, а None - рекомендация браузеру поступить на своё усмотрение
    if samesite not in ('Strict', 'Lax', 'None', None):
        raise ValueError("invalid SameSite value; must be 'Strict', 'Lax', 'None' or None")

    # здесь был _cookie_quote, но я его убрал для совместимости с текущим поведением
    buf = [key + b'=' + value]

    # XXX: In theory all of these parameters that are not marked with `None`
    # should be quoted.  Because stdlib did not quote it before I did not
    # want to introduce quoting there now.
    for k, v, q in ((b'Domain', domain, True),
                    (b'Expires', expires, False),
                    (b'Max-Age', max_age, False),
                    (b'Secure', secure, None),
                    (b'HttpOnly', httponly, None),
                    (b'Path', path, False),
                    (b'SameSite', samesite, False)):
        if q is None:
            if v:
                buf.append(k)
            continue

        if v is None:
            continue

        if k == b'Expires' and not add_expires:
            continue

        tmp = bytearray(k)
        if not isinstance(v, (bytes, bytearray)):
            v = to_bytes(text_type(v), charset)
        if q:
            v = _cookie_quote(v)
        tmp += b'=' + v
        buf.append(bytes(tmp))

    # The return value will be an incorrectly encoded latin1 header on
    # Python 3 for consistency with the headers object and a bytestring
    # on Python 2 because that's how the API makes more sense.
    rv = b'; '.join(buf)
    if not six.PY2:
        rv = rv.decode('latin1')

    # Warn if the final value of the cookie is less than the limit. If the
    # cookie is too large, then it may be silently ignored, which can be quite
    # hard to debug.
    cookie_size = len(rv)

    if max_size and cookie_size > max_size:
        value_size = len(value)
        warnings.warn(
            'The "{key}" cookie is too large: the value was {value_size} bytes'
            ' but the header required {extra_size} extra bytes. The final size'
            ' was {cookie_size} bytes but the limit is {max_size} bytes.'
            ' Browsers may silently ignore cookies larger than this.'.format(
                key=key,
                value_size=value_size,
                extra_size=cookie_size - value_size,
                cookie_size=cookie_size,
                max_size=max_size
            ),
            stacklevel=2
        )

    return rv


# Код парсинга кук портирован из werkzeug версии 0.10.4
# В текущей версии не парсятся куки, выставленные нами же, если они содержат юникод (из-за
# того, что мы не оборачиваем значение с недопустимыми символами в куке в кавычки)
# На новую версию не хотим переходить, т.к. сильно изменился формат кодирования кук
# История описана в тикете PASSP-12573
# Отличия от werkzeug: text_type заменён на unicode
def parse_cookie(header, charset='utf-8', errors='replace', cls=None):
    """Parse a cookie.  Either from a string or WSGI environ.
    Per default encoding errors are ignored.  If you want a different behavior
    you can set `errors` to ``'replace'`` or ``'strict'``.  In strict mode a
    :exc:`HTTPUnicodeError` is raised.
    .. versionchanged:: 0.5
       This function now returns a :class:`TypeConversionDict` instead of a
       regular dict.  The `cls` parameter was added.
    :param header: the header to be used to parse the cookie.  Alternatively
                   this can be a WSGI environment.
    :param charset: the charset for the cookie values.
    :param errors: the error behavior for the charset decoding.
    :param cls: an optional dict class to use.  If this is not specified
                       or `None` the default :class:`TypeConversionDict` is
                       used.
    """
    if isinstance(header, dict):
        header = header.get('HTTP_COOKIE', '')
    elif header is None:
        header = ''

    # If the value is an unicode string it's mangled through latin1.  This
    # is done because on PEP 3333 on Python 3 all headers are assumed latin1
    # which however is incorrect for cookies, which are sent in page encoding.
    # As a result we

    # Здесь text_type был заменён на unicode
    if isinstance(header, six.text_type):
        header = header.encode('latin1', 'replace')

    if cls is None:
        cls = TypeConversionDict

    def _parse_pairs():
        for key, val in _cookie_parse_impl(header):
            key = to_unicode(key, charset, errors, allow_none_charset=True)
            val = to_unicode(val, charset, errors, allow_none_charset=True)
            yield try_coerce_native(key), val

    # Реверсим список кук для согласованности с фронтендом
    # (https://st.yandex-team.ru/PASSP-15163#1474299949000)
    return cls(reversed(list(_parse_pairs())))


def fix_simple_coookie_samesite_issue29613():
    # https://bugs.python.org/issue29613
    from six.moves.http_cookies import Morsel
    Morsel._reserved[str('samesite')] = str('SameSite')


class TypeConversionDict(dict):
    """Works like a regular dict but the :meth:`get` method can perform
    type conversions.  :class:`MultiDict` and :class:`CombinedMultiDict`
    are subclasses of this class and provide the same feature.
    .. versionadded:: 0.5
    """

    def get(self, key, default=None, type=None):
        """Return the default value if the requested data doesn't exist.
        If `type` is provided and is a callable it should convert the value,
        return it or raise a :exc:`ValueError` if that is not possible.  In
        this case the function will return the default as if the value was not
        found:
        >>> d = TypeConversionDict(foo='42', bar='blub')
        >>> d.get('foo', type=int)
        42
        >>> d.get('bar', -1, type=int)
        -1
        :param key: The key to be looked up.
        :param default: The default value to be returned if the key can't
                        be looked up.  If not further specified `None` is
                        returned.
        :param type: A callable that is used to cast the value in the
                     :class:`MultiDict`.  If a :exc:`ValueError` is raised
                     by this callable the default value is returned.
        """
        try:
            rv = self[key]
            if type is not None:
                rv = type(rv)
        except (KeyError, ValueError):
            rv = default
        return rv
