# -*- coding: utf-8 -*-

import logging
import time

from passport.backend.core.conf import settings
from passport.backend.core.grants.grants_config import (
    check_ip_counters_always_empty,
    check_ip_counters_always_full,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.redis_manager import redis_manager


log = logging.getLogger(__name__)


KEY_PLACEHOLDER = '_'


class Buckets(object):
    """Механизм счетчиков на основе корзин.

    Вычисляет сумму за последние `count` периодов длиной `duration` секунд.

    В redis'е хранится `count` ключей, время жизни которых устанавливается в
    `duration * count`.

    Идентификатор текущей корзины строится из префикса `prefix`, ключа и
    времени. Время при этом целочисленно делится на `duration`, и мы получаем
    новый идентификатор каждые `duration` секунд.

    В качестве суммы берется сумма последних `count` ключей включая текущий.
    """

    def __init__(self, prefix, count, duration, limit, redis=None):
        self.prefix = prefix
        self.count = count
        self.duration = duration
        self.limit = limit
        if redis is None:
            redis = get_counters_redis_manager()
        self.redis = redis

    def _id(self, key, ts):
        return '%s:%s:%d' % (self.prefix, key, ts // self.duration)

    def incr(self, key=KEY_PLACEHOLDER):
        """Увеличивает значение в текущей корзине.

        Возвращает новое значение ТЕКУЩЕЙ(!) корзины. Может использоваться для
        "оптимизации" если текущая корзина переполняет счетчик сама по себе. Но
        для полной проверки необходимо использовать метод get.
        """

        bucket_id = self._id(key, time.time())

        try:
            ret = self.redis.incr(bucket_id)
            self.redis.expire(bucket_id, self.duration * self.count)
        except redis_manager.RedisError:
            log.warn("Can't increment counter for bucket %s", bucket_id)
            ret = 0

        return ret

    def get(self, key=KEY_PLACEHOLDER):

        ts = time.time()
        ids = []
        for i in range(self.count):
            ids.append(self._id(key, ts))
            ts -= self.duration

        try:
            buckets = [int(item) for item in self.redis.mget(ids) if item]
        except redis_manager.RedisError:
            log.warning("Can't get counters for %s", key)
            buckets = [0]
        except ValueError:
            log.warning("Bad value in counters for %s", key)
            buckets = [0]

        return sum(buckets)

    def hit_limit(self, key=KEY_PLACEHOLDER, limit=None):
        """
        Счётчик достиг или превысил лимит.
        Лимит по умолчанию задается в конфигурации счетчика, также можно передать другое значение лимита.
        """
        return self.get(key) >= (limit or self.limit)

    def hit_limit_by_ip(self, key=KEY_PLACEHOLDER, user_ip=None, limit=None):
        if user_ip is None:
            user_ip = key

        if check_ip_counters_always_empty(user_ip):
            return False
        if check_ip_counters_always_full(user_ip):
            return True

        return self.hit_limit(key=key, limit=limit)


def get_counters_redis_manager():
    return redis_manager.get_redis_mgr(
        settings.COUNTERS_REDIS_MANAGERS[get_current_host().get_dc()]
    )


def get_buckets(prefix, redis=None):
    try:
        count, duration, limit = settings.COUNTERS[prefix]
    except KeyError:
        raise ValueError('Counter "%s" is not configured!' % prefix)
    return Buckets(prefix, count, duration, limit, redis)


def do_buckets_exist(prefix):
    return prefix in settings.COUNTERS
