# -*- coding: utf-8 -*-

from collections import (
    defaultdict,
    namedtuple,
)

from passport.backend.core.builders.kolmogor.exceptions import (
    KolmogorPermanentError,
    KolmogorTemporaryError,
)


def _group_by_key_space(counters):
    groups = defaultdict(list)
    for counter in counters:
        groups[counter.key_space].append(counter)

    # Соберём счётчики в группы, сохранив порядок групп (учитывается только
    # порядок первых представителей каждой группы)
    seen = set()
    for counter in counters:
        if counter.key_space not in seen:
            seen.add(counter.key_space)
            yield counter.key_space, groups[counter.key_space]


class Counter(namedtuple('Counter', 'key_space name limit')):
    @property
    def fqn(self):
        return self.key_space + '.' + self.name


class KolmogorToolkit(object):
    def __init__(self, kolmogor):
        self.kolmogor = kolmogor

    def check_counters(self, counters):
        for _, counters in _group_by_key_space(counters):
            self._check_key_space(counters)

    def failsafe_check_counters(self, counters):
        try:
            self.check_counters(counters)
            return True
        except BackendUnavailableCounterError:
            return False

    def inc_counters(self, counters):
        for _, counters in _group_by_key_space(counters):
            self._inc_key_space(counters)

    def failsafe_inc_counters(self, counters):
        try:
            self.inc_counters(counters)
            return True
        except BackendUnavailableCounterError:
            return False

    def _check_key_space(self, counters):
        """
        Проверяет счётчики одного пространства.
        """
        counters = list(counters)
        if not counters:
            return
        try:
            values = self.kolmogor.get(
                counters[0].key_space,
                [c.name for c in counters],
            )
        except (KolmogorTemporaryError, KolmogorPermanentError):
            raise BackendUnavailableCounterError()
        overflow = list()
        for counter in counters:
            value = values.get(counter.name, 0)
            if value >= counter.limit:
                overflow.append((counter, value))
        if overflow:
            raise OverflowCounterError(overflow)

    def _inc_key_space(self, counters):
        """
        Увеличивает счётчики одного пространства.
        """
        counters = list(counters)
        if not counters:
            return
        try:
            self.kolmogor.inc(
                counters[0].key_space,
                [c.name for c in counters],
            )
        except (KolmogorTemporaryError, KolmogorPermanentError):
            raise BackendUnavailableCounterError()


class CounterError(Exception):
    pass


class OverflowCounterError(CounterError):
    def __init__(self, overflow_list):
        super(OverflowCounterError, self).__init__(overflow_list)
        self.overflow_list = overflow_list


class BackendUnavailableCounterError(CounterError):
    pass
