# -*- coding: utf-8 -*-

import logging
from os import path
import time

from passport.backend.utils.file import read_file


log = logging.getLogger('passport.crypto')


class BaseKeyStorage(object):
    def __init__(self):
        self._KEY_CACHE = {}

    def get_key(self, key_number=None):
        if key_number is None:
            key_number = self._get_key_number()

        if key_number not in self._KEY_CACHE:
            key_path = self._get_key_filepath(key_number)
            key_data = read_file(key_path)
            self._KEY_CACHE[key_number] = key_data

            if not self.is_key_available(key_number + 1):
                next_key_path = self._get_key_filepath(key_number + 1)
                log.error('Next epoch key does not exist, provide new keys ASAP!: %s' % next_key_path)  # pragma: no cover

        return self._KEY_CACHE[key_number], key_number

    def is_key_available(self, key_number):
        key_path = self._get_key_filepath(key_number)
        return path.exists(key_path)

    def delete_key_from_cache(self, key_number):
        self._KEY_CACHE.pop(key_number, None)

    def _get_key_number(self):
        timestamp = int(time.time())
        return timestamp / self._epoch_length

    @property
    def _epoch_length(self):
        raise NotImplementedError()  # pragma: no cover

    def _get_key_filepath(self, key_number):
        return path.join(self._key_dirpath, '%d.key' % key_number)

    @property
    def _key_dirpath(self):
        raise NotImplementedError()  # pragma: no cover


class SigningKeyStorage(BaseKeyStorage):
    def __init__(self, dirpath):
        super(SigningKeyStorage, self).__init__()
        self._dirpath = dirpath

    @property
    def _key_dirpath(self):
        return self._dirpath
