# -*- coding: utf-8 -*-

import base64
from hashlib import md5
import logging

from cryptography.exceptions import (
    InvalidKey,
    InvalidSignature,
    InvalidTag,
)
from passport.backend.core.crypto import (
    aes_gcm,
    md5crypt,
)
from passport.backend.core.crypto.exceptions import DecryptionError
from passport.backend.utils.string import (
    compare_strings,
    smart_bytes,
)


log = logging.getLogger('passport.crypto')


def hash_string(string, salt=None):
    return md5crypt.md5crypt(string.encode('utf-8'), salt)


def dumb_hash_string(string):
    return md5(smart_bytes(string)).hexdigest()


def check_hashed_string(string, hashed_string):
    """Проверяет, совпадает ли переданная строка с хешем."""
    if not hashed_string:
        return False

    # FIXME: Выпилить этот жуткий хак, после того как научимся получить от ЧЯ
    # версию парольного хэша.
    if len(hashed_string) == 32:
        return compare_strings(dumb_hash_string(string), hashed_string)
    else:
        _, method, salt, _ = hashed_string.split(b'$')

        if method != b'1':
            raise ValueError('Unknown hash type')

        hashed_string_1 = hash_string(string, salt).decode('latin1')
        hashed_string_2 = hashed_string.decode('latin1')
        return compare_strings(hashed_string_1, hashed_string_2)


def simple_encrypt(key_storage, value):
    """
    Зашифровать значение для записи в лог (рекомендую передавать сюда байтовую
    строку, а не текст).
    """
    key, key_number = key_storage.get_key()

    associated_data = b'key_number=%d' % key_number
    iv, ciphertext, auth_tag = aes_gcm.encrypt(key, smart_bytes(value), associated_data)

    return b':'.join([
        base64.b64encode(iv),
        base64.b64encode(ciphertext),
        associated_data,
        base64.b64encode(auth_tag),
    ])


def simple_decrypt(key_storage, value):
    """
    Расшифровать значение из лога. Отдает байтовую строку (не обязательно
    валидный utf-8).
    """
    try:
        iv, ciphertext, associated_data, auth_tag = value.split(b':')
    except ValueError as e:
        log.error('Failed to split value', exc_info=e)
        raise DecryptionError()

    try:
        iv = base64.b64decode(iv)
        ciphertext = base64.b64decode(ciphertext)
        auth_tag = base64.b64decode(auth_tag)
    except TypeError as e:
        log.error('Failed to decode value', exc_info=e)
        raise DecryptionError()

    try:
        _, key_number = associated_data.split(b'=')
        key_number = int(key_number)
    except ValueError as e:
        log.error('Failed to parse key number', exc_info=e)
        raise DecryptionError()

    try:
        key, _ = key_storage.get_key(key_number)
    except IOError as e:
        log.error('Cannot read key number %s', key_number, exc_info=e)
        raise DecryptionError()

    try:
        return aes_gcm.decrypt(key, associated_data, iv, ciphertext, auth_tag)
    except (InvalidKey, InvalidTag, InvalidSignature) as e:
        log.error('Decryption failed', exc_info=e)
        raise DecryptionError()
