# -*- coding: utf-8 -*-
import logging


log = logging.getLogger('passport.db.query')


class DbQuery(object):
    _SHARDED = False
    _IGNORE_ERRORS = False

    def is_sharded(self):
        return self._SHARDED

    def ignore_errors(self):
        return self._IGNORE_ERRORS

    def to_query(self):
        raise NotImplementedError()  # pragma: no cover

    def get_table(self):
        raise NotImplementedError()  # pragma: no cover

    def query_params(self):
        return dict()  # pragma: no cover


class DbQueryForUID(DbQuery):
    def __init__(self, uid):
        self.uid = uid

    def __eq__(self, other):
        if self.__class__ != other.__class__:
            return False
        return self.uid == other.uid and self.query_params() == other.query_params()

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        return '<%s: uid: %s | params: %s>' % (self.__class__.__name__, self.uid, self.query_params())


class DbTransaction(object):
    """
    Класс-декоратор для сериализатора, запросы которого должны выполняться в рамках одной транзакции.
    В сериализаторе должна быть предусмотрена возможность повторного вызова,
    что необходимо для ретраев транзакции.
    """
    def __init__(self, serializer):
        self.serializer = serializer

    def __call__(self, *args, **kwargs):
        """
        При вызове сериализатора - создаем контейнер, специфичный для переданных аргументов,
        сериализацию откладываем.
        """
        return DbTransactionContainer(self.serializer, args, kwargs)


class DbTransactionContainer(object):
    def __init__(self, serializer, args, kwargs):
        self.serializer = serializer
        self.args = args
        self.kwargs = kwargs

    def get_queries(self):
        """
        Получить запросы для выполнения в БД. При необходимости повтора транзакции может вызываться повторно.
        """
        for query in self.serializer(*self.args, **self.kwargs):
            yield query


class QueryJoiner(object):
    def __init__(self):
        self.reset()

    def add_query(self, query, callback):
        """
        Добавить запрос в очередь на исполнение, при возможности склеить запросы.
        Возвращает признак необходимости выполнения накопившихся запросов.
        """
        # Если запрос не джойнится, то просто записываем его в очередь выполнения
        can_be_joined = getattr(query, 'can_be_joined', False)
        if not can_be_joined:
            self.queries.append((query, callback))
        else:
            cls = query.__class__
            # Достаем индекс запроса в executed_order
            query_index = self.lookup.get(cls)
            # Новый запрос
            if query_index is None:
                self.queries.append((query, callback))
                self.lookup[cls] = len(self.queries) - 1
            else:
                old_query, _callback = self.queries[query_index]
                # При использовании join_queries, _callback всегда равен None
                self.queries[query_index] = (old_query + query, callback)
                if callback:
                    # Если встретили callback - нельзя менять порядок выполнения запросов. Учитывая, что мы
                    # уже склеили запрос - переложим его в конец очереди.
                    query, callback = self.queries.pop(query_index)
                    self.queries.append((query, callback))
                    self.lookup[cls] = len(self.queries) - 1
        return bool(callback) or isinstance(query, DbTransactionContainer)

    def reset(self):
        self.queries = []
        self.lookup = {}


def split_query_and_callback(pair):
    try:
        query, callback = pair
    except (ValueError, TypeError):
        query = pair
        callback = None
    return query, callback


def join_queries(queries):
    joiner = QueryJoiner()

    for pair in queries:
        query, callback = split_query_and_callback(pair)
        need_execute = joiner.add_query(query, callback)
        if need_execute:
            for query, callback in joiner.queries:
                yield query, callback
            joiner.reset()
    for query, callback in joiner.queries:
        yield query, callback
