# -*- coding: utf-8 -*-

import logging
import re

from passport.backend.core.conf import settings
from passport.backend.core.db.query import (
    DbQuery,
    DbQueryForUID,
)
from passport.backend.core.db.read_api.exceptions import UidsByRemovedAliasesLoginError
from passport.backend.core.db.read_api.runner import EavReadingActionRunner
from passport.backend.core.db.schemas import removed_aliases_table
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE as ATT
from passport.backend.core.types.login.login import (
    login_is_phonish,
    login_is_social,
    normalize_login,
)
import six
from sqlalchemy.sql import select
from sqlalchemy.sql.expression import (
    and_,
    or_,
)


log = logging.getLogger('passport.db.read_api.search')

REMOVED_ALIASES_SEARCH_LIMIT = 5000

PHONENUMBER_ALIAS_RE = re.compile(r'^[\d+()-]+')

ALIAS_TYPE_TO_NAME = {v: k for k, v in ATT.items()}

ALT_DOMAIN_ID_TO_NAME = {v: k for k, v in settings.ALT_DOMAINS.items()}


class UidsByRemovedAliasesQuery(DbQuery):
    def query_params(self):
        return self.login, self.limit

    def get_table(self):
        return removed_aliases_table

    def __init__(self, login, limit=REMOVED_ALIASES_SEARCH_LIMIT):
        self.login = login
        self.limit = limit

    def to_query(self):
        t = removed_aliases_table
        where_clauses = _build_where_clauses(self.login)
        if not where_clauses:
            raise UidsByRemovedAliasesLoginError()
        return select([t.c.uid]).distinct().where(or_(*where_clauses)).limit(self.limit)

    def parse_query_result(self, query_result):
        return [uid_tuple[0] for uid_tuple in query_result.fetchall()]

    def __repr__(self):
        return '<%s: login: %s | limit: %s>' % (self.__class__.__name__, self.login, self.limit)


class RemovedAliasesByUidQuery(DbQueryForUID):
    def query_params(self):
        return None

    def get_table(self):
        return removed_aliases_table

    def to_query(self):
        t = removed_aliases_table
        return select([t.c.value, t.c.type]).distinct().where(t.c.uid == self.uid)

    def parse_query_result(self, query_result):
        aliases_by_type = {}
        for raw_alias, alias_type in query_result.fetchall():
            alias = raw_alias.decode('utf8')
            alias_type = ALIAS_TYPE_TO_NAME.get(alias_type)
            if alias_type == 'altdomain':
                domain_id, _, login = alias.partition('/')
                alias = login + '@' + ALT_DOMAIN_ID_TO_NAME[int(domain_id)]
            elif alias_type in ('pdd', 'pddalias'):
                domain, _, login = alias.partition('/')
                alias = '@'.join([login, domain.encode('utf8').decode('idna')])
            elif alias_type == 'lite':
                login, _, domain = alias.partition('@')
                alias = '@'.join([login, domain.encode('utf8').decode('idna')])

            aliases_by_type.setdefault(alias_type, []).append(alias)
        return aliases_by_type

    def __repr__(self):
        return '<%s: uid: %s>' % (self.__class__.__name__, self.uid)


def _parse_login(login):
    login_part, _, domain = login.partition('@')

    external_domain_id = settings.ALT_DOMAINS.get(domain)

    if domain in settings.NATIVE_EMAIL_DOMAINS:
        domain = ''

    return login_part, domain, external_domain_id


def _build_aliases_clause(aliases, value):
    t = removed_aliases_table
    if len(aliases) == 1:
        type_expression = t.c.type == ATT[aliases[0]]
    else:
        type_expression = t.c.type.in_([ATT[alias] for alias in aliases])
    return and_(
        type_expression,
        t.c.value == value.encode('utf8'),
    )


def _build_where_clauses(login):
    login_part, domain, external_domain_id = _parse_login(login)

    where_clauses = []

    if login_part:
        normalized_login = normalize_login(login_part)

        if external_domain_id:
            where_clauses.append(
                _build_aliases_clause(
                    ['altdomain'],
                    '/'.join((str(external_domain_id), login_part)),
                ),
            )
        elif domain:
            where_clauses.extend([
                _build_aliases_clause(
                    ['pdd', 'pddalias'],
                    '/'.join((
                        domain.encode('idna').decode('utf8'),
                        login_part,
                    )),
                ),
                _build_aliases_clause(
                    ['lite'],
                    '@'.join((
                        login_part,
                        domain.encode('idna').decode('utf8'),
                    )),
                ),
            ])
        else:
            if login_is_social(normalized_login):
                where_clauses.append(
                    _build_aliases_clause(['social'], normalized_login),
                )
            elif login_is_phonish(normalized_login):
                where_clauses.append(
                    _build_aliases_clause(['phonish'], normalized_login),
                )
            elif PHONENUMBER_ALIAS_RE.match(normalized_login):
                where_clauses.append(
                    _build_aliases_clause(
                        ['phonenumber'],
                        u''.join([c for c in normalized_login if six.text_type.isdigit(c)]),
                    ),
                )
            else:
                where_clauses.append(
                    _build_aliases_clause(['portal', 'mail', 'narodmail'], normalized_login),
                )
    return where_clauses


def find_uids_by_removed_aliases(login):
    return EavReadingActionRunner().execute(UidsByRemovedAliasesQuery(login))


def find_removed_aliases_by_uid(uid):
    return EavReadingActionRunner().execute(RemovedAliasesByUidQuery(uid))
