# -*- coding: utf-8 -*-

from passport.backend.core.db.types import TIMESTAMP
from passport.backend.core.types.account.account import PDD_UID_BOUNDARY
from sqlalchemy import (
    DDL,
    event,
)
from sqlalchemy.schema import (
    Column,
    Index,
    MetaData,
    Table,
)
from sqlalchemy.sql.functions import current_timestamp
from sqlalchemy.types import (
    BigInteger,
    BINARY,
    Boolean,
    Integer,
    LargeBinary,
    SmallInteger,
    VARBINARY,
    VARCHAR,
)


central_metadata = MetaData()
shard_metadata = MetaData()


aliases_table = Table(
    'aliases',
    central_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('type', SmallInteger, nullable=False, default=0),
    Column('value', VARBINARY(255), nullable=False, default=b''),
    Column('surrogate_type', VARBINARY(255), primary_key=True, autoincrement=False, nullable=False, default=b''),
    Index('type', 'type', 'value', unique=True),
)


suid2_table = Table(
    'suid2',
    central_metadata,
    Column('suid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('uid', BigInteger, nullable=False, default=0),
    Index('suid2_uid', 'uid', unique=True),
)


attributes_table = Table(
    'attributes',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('type', SmallInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('value', VARBINARY(65523), nullable=False, default=''),
    Index('type', 'type'),
)


password_history_eav_table = Table(
    'password_history',
    shard_metadata,
    Column('ts', TIMESTAMP(), server_default=current_timestamp(), nullable=False),
    Column('reason', SmallInteger, nullable=False, server_default='1'),
    Column('uid', BigInteger, nullable=False, server_default='0'),
    Column('encrypted_password', VARBINARY(255), nullable=False, server_default=''),
    Index('pwdhist_uid', 'uid'),
)


removed_aliases_table = Table(
    'removed_aliases',
    central_metadata,
    Column('uid', BigInteger, nullable=False, default=0),
    Column('type', SmallInteger, nullable=False, default=0),
    Column('value', VARBINARY(255), primary_key=True, autoincrement=False, nullable=False, default=''),
    Index('alias_type', 'value', 'uid', unique=True),
    Index('alias_uid', 'uid'),
)


# uid в базе MySQL - BIGINT. Влияет на создание
# таблиц, которым мы пользуемся только для SQLite,
# который умеет autoincrement только с Integer
uid_table = Table(
    'uid',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Index('incr_uid', 'id', unique=True),
    sqlite_autoincrement=True,
)


pdduid_table = Table(
    'pdduid',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Index('incr_pdduid', 'id', unique=True),
    sqlite_autoincrement=True,
)


suid_table = Table(
    'suid',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Index('incr_suid', 'id', unique=True),
    sqlite_autoincrement=True,
)


pddsuid_table = Table(
    'pddsuid',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Index('incr_pddsuid', 'id', unique=True),
    sqlite_autoincrement=True,
)

totp_secret_id_table = Table(
    'totp_secret_id',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Index('incr_totp_secret', 'id', unique=True),
    sqlite_autoincrement=True,
)

webauthn_credential_id_table = Table(
    'webauthn_credential_id',
    central_metadata,
    # На самом деле ID имеет тип BigInteger, но т.к. SQLite поддерживает autoincrement
    # только для Integer-столбцов, пишем тут Integer - актуально для тестов.
    Column('webauthn_credential_id', Integer, primary_key=True, nullable=False, autoincrement=True),
    sqlite_autoincrement=True,
)

phone_id_table = Table(
    'phone_id',
    central_metadata,
    # На самом деле ID имеет тип BigInteger, но т.к. SQLite поддерживает autoincrement
    # только для Integer-столбцов, пишем тут Integer - актуально для тестов.
    Column('phone_id', Integer, primary_key=True, nullable=False, autoincrement=True),
    sqlite_autoincrement=True,
)

email_id_table = Table(
    'email_id',
    central_metadata,
    # На самом деле ID имеет тип BigInteger, но т.к. SQLite поддерживает autoincrement
    # только для Integer-столбцов, пишем тут Integer - актуально для тестов.
    Column('email_id', Integer, primary_key=True, nullable=False, autoincrement=True),
    sqlite_autoincrement=True,
)


webauthn_credentials_table = Table(
    'webauthn_credentials',
    central_metadata,
    Column('uid', BigInteger, nullable=False),
    Column('credential_id', VARBINARY(255), primary_key=True, nullable=False),
    Index('webauthn_credential_bindings_uid_credential_id', 'uid', 'credential_id', unique=True),
)


email_bindings_table = Table(
    'email_bindings',
    shard_metadata,
    Column('uid', BigInteger, nullable=False, default=0, index=True, autoincrement=False),
    Column('address', VARBINARY(length=500), nullable=False, autoincrement=False),
    Column('email_id', BigInteger, nullable=False, default=0),
    Column('bound', TIMESTAMP(), server_default=current_timestamp(), nullable=False),
    Index('binding_uid', 'address', 'uid', unique=True),
    Index('binding_email_id', 'email_id', unique=True),
)


phone_bindings_table = Table(
    'phone_bindings',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('number', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('phone_id', BigInteger, nullable=False, default=0),
    Column('bound', TIMESTAMP(), server_default=current_timestamp(), nullable=False),
    Column('flags', SmallInteger, nullable=False, default=0),
    Index('phone_id', 'phone_id', unique=True),
)


phone_bindings_history_table = Table(
    'phone_bindings_history',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('number', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('bound', TIMESTAMP(), primary_key=True, autoincrement=False, server_default=current_timestamp(), nullable=False),
    Index('bound', 'bound', unique=False),
)


phone_operations_table = Table(
    'phone_operations',
    shard_metadata,
    Column('id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('uid', BigInteger, nullable=False, default=0),
    Column('phone_id', BigInteger, nullable=False, default=0),
    Column('security_identity', BigInteger, nullable=False, default=0),
    Column('type', SmallInteger, nullable=False, default=0),
    Column('started', TIMESTAMP(), nullable=False, server_default=current_timestamp()),
    Column('finished', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('code_value', VARCHAR(length=100), nullable=False, default=''),
    Column('code_checks_count', SmallInteger, nullable=False, default=0),
    Column('code_send_count', SmallInteger, nullable=False, default=0),
    Column('code_last_sent', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('code_confirmed', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('password_verified', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('flags', SmallInteger, nullable=False, default=0),
    Column('phone_id2', BigInteger, nullable=False, default=0),
    Index('uid', 'uid', 'phone_id', unique=True),
    Index('security', 'uid', 'security_identity', unique=True),
    Index('finished', 'finished', 'phone_id', unique=False),
    Index('incr_id', 'id', unique=True),
    sqlite_autoincrement=True,
)


extended_attributes_table = Table(
    'extended_attributes',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('entity_type', SmallInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('entity_id', BigInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('type', SmallInteger, primary_key=True, autoincrement=False, nullable=False, default=0),
    Column('value', VARBINARY(length=65514), nullable=False),
    Index('all_ids', 'uid', 'entity_type', 'entity_id', 'type', unique=True),
)


pdd_domains_table = Table(
    'domains',
    central_metadata,
    Column('domain_id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('master_domain_id', Integer, nullable=False, default=0),
    Column('name', VARBINARY(length=255), nullable=False),
    Column('enabled', Boolean, nullable=False, default=False),
    Column('mx', Boolean, nullable=False, default=False),
    Column('ts', TIMESTAMP(), nullable=False, server_default='0000-00-00 00:00:00'),
    Column('admin_uid', BigInteger, nullable=False, default=0),
    Column('default_uid', BigInteger, nullable=False, default=0),
    Column('options', VARBINARY(length=255), nullable=False),
    Index('domain_id', 'domain_id', unique=True),
    Index('name', 'name', unique=True),
    Index('admin_uid', 'admin_uid'),
    Index('master_domain_id', 'master_domain_id'),
)


reserved_logins_table = Table(
    'reserved_logins',
    central_metadata,
    Column('login', VARBINARY(length=255), primary_key=True, autoincrement=False, nullable=False),
    Column('free_ts', TIMESTAMP(), nullable=False),
    Index('free_ts', 'free_ts'),
)


hosts_table = Table(
    'hosts',
    central_metadata,
    # В БД это поле smallint, но здесь использую Integer чтобы работал autoincrement
    Column('host_id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('host_ip', VARCHAR(length=16)),
    Column('host_name', VARCHAR(length=255)),
    Column('db_id', VARCHAR(length=40)),
    # В БД следующие три поля tinyint, но для нас нет разницы
    Column('sid', SmallInteger, nullable=False, default=0),
    Column('prio', SmallInteger, nullable=False, default=0),
    Column('host_number', SmallInteger, nullable=False, default=0),
    Column('mx', VARCHAR(length=30)),
    Index('db_id', 'db_id'),
)


domains_hosts_table = Table(
    'domains_hosts',
    central_metadata,
    # В БД это поле smallint, но здесь использую Integer чтобы работал autoincrement
    Column('host_id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('host_ip', VARCHAR(length=16)),
    Column('host_name', VARCHAR(length=255)),
    Column('db_id', VARCHAR(length=40)),
    # В БД следующие три поля tinyint, но для нас нет разницы
    Column('sid', SmallInteger, nullable=False, default=0),
    Column('prio', SmallInteger, nullable=False, default=0),
    Column('host_number', SmallInteger, nullable=False, default=0),
    Column('mx', VARCHAR(length=30)),
    Index('domain_db_id', 'db_id'),
)


domains_events_table = Table(
    'domains_events',
    central_metadata,
    Column('id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('domain_id', Integer, nullable=False, server_default='0'),
    Column('meta', VARBINARY(length=4096)),
    Column('type', SmallInteger, nullable=False, server_default='0'),
    Column('ts', TIMESTAMP(), server_default=current_timestamp(), nullable=False),
)


tracks_table = Table(
    'tracks',
    shard_metadata,
    # На самом деле ID имеет тип BigInteger, но т.к. SQLite поддерживает autoincrement
    # только для Integer-столбцов, пишем тут Integer - актуально для тестов.
    Column('id', Integer, primary_key=True, autoincrement=True, nullable=False),
    Column('uid', BigInteger, nullable=False, server_default='0'),
    Column('track_id', BINARY(length=32), nullable=False, server_default=''),
    Column('created', TIMESTAMP, nullable=False, server_default=current_timestamp()),
    Column('expired', TIMESTAMP, nullable=False, server_default='0'),
    Column('content', VARBINARY(length=65477), nullable=False, server_default=''),
    Index('tracks_track_id', 'track_id', unique=True),
    Index('tracks_uid', 'uid'),
    Index('tracks_expired', 'expired'),
)


yakey_backups_table = Table(
    'yakey_backups',
    central_metadata,
    Column('phone_number', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    Column('backup', LargeBinary(), nullable=False),
    Column('updated', TIMESTAMP(), nullable=False),
    Column('device_name', VARCHAR(255)),
    Index('backup_updated', 'updated'),
)


account_deletion_operations_table = Table(
    'account_deletion_operations',
    shard_metadata,
    # В БД нужно использовать BigInteger, здесь используется Integer из-за
    # проблем sqlite.
    Column('deletion_id', Integer, primary_key=True, autoincrement=True),
    Column('uid', BigInteger, nullable=False, unique=True),
    Column('started_at', TIMESTAMP(), nullable=False, index=True),
)


passman_recovery_keys_table = Table(
    'passman_recovery_keys',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    Column('key_id', VARBINARY(length=256), primary_key=True, autoincrement=False, nullable=False),
    Column('recovery_key', VARBINARY(length=256), nullable=False),
)


device_public_key_table = Table(
    'device_public_key',
    central_metadata,
    # Идентификатор устройства, в котором лежит соответствующий закрытый ключ
    Column('device_id', VARBINARY(2048), primary_key=True, nullable=False),
    # Владелец ключа; в Грантушке заведены соответствующие гранты на создание,
    # удаление и обновления ключей, связанных с этим владельцем.
    Column('owner_id', SmallInteger, nullable=False),
    # Открытый ключ, которым можно проверить текст, подписанный устройством
    Column('public_key', VARBINARY(60000), nullable=False),
    # Версия определяет, какой алгоритм или его модификация должна
    # использоваться для проверки подписи.
    Column('version', SmallInteger, nullable=False),
)


family_info_table = Table(
    'family_info',
    central_metadata,
    # Идентификатор семьи (в таблице bigint, integer для sqlite-совместимости)
    Column('family_id', Integer, primary_key=True, autoincrement=True, nullable=False),
    # ID админа семьи
    Column('admin_uid', BigInteger, nullable=False),
    # Метаданные семьи
    Column('meta', VARBINARY(32768), nullable=False),
    # Уникальный индекс по админу
    Index('family_info_admin_id', 'admin_uid', unique=True),
    sqlite_autoincrement=True,
)


family_members_table = Table(
    'family_members',
    central_metadata,
    # Идентификатор семьи
    Column('family_id', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    # ID члена семьи
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    # Занимаемое место в семье - уникальное для family_id число
    Column('place', SmallInteger, autoincrement=False, nullable=False),
    # Уникальный индекс по месту в семье
    Index('family_members_family_id_place', 'family_id', 'place', unique=True),
)

phone_bindings_history_delete_tasks_table = Table(
    'phone_bindings_history_delete_tasks',
    central_metadata,
    # ID задания, PK (в таблице bigint, integer для sqlite-совместимости)
    Column('task_id', Integer, primary_key=True, autoincrement=True, nullable=False),
    # UID
    Column('uid', BigInteger, index=True, nullable=False),
    # Время удаления аккаунта
    Column('deletion_started_at', TIMESTAMP, index=True, nullable=False),
    sqlite_autoincrement=True,
)


# Т.к. по умолчанию ключи в таблицах SQLite начинаются отсчитывать с нуля,
# нам необходимо вручную "сдвинуть" их на минимальное значение UID'а
# для ПДДшника.
for table, incr_start in (
    (pdduid_table, PDD_UID_BOUNDARY),
    (pddsuid_table, PDD_UID_BOUNDARY),
):
    event.listen(
        table,
        'after_create',
        DDL(
            'INSERT INTO sqlite_sequence (seq, name) VALUES (%d, \'%s\')' % (incr_start, table.fullname),
        ).execute_if(dialect=['sqlite']),
    )
