# -*- coding: utf-8 -*-
import six


def encode_params_for_db(params):
    return {
        key: value.encode('utf8') if isinstance(value, six.text_type) else value
        for key, value in params.items()
    }


def with_ignore_prefix(expression):
    """
    Добавляем к запросу префикс IGNORE

    Работает для бэкендов: sqlite, mysql
    """
    return (expression
            .prefix_with('IGNORE', dialect='mysql')
            .prefix_with('OR IGNORE', dialect='sqlite'))


def insert_with_on_duplicate_key_update(table, keys, binary_value_ids=None):
    return table.insert(mysql_on_duplicate_update_keys=sorted(keys), mysql_binary_value_ids=binary_value_ids)


def insert_with_on_duplicate_key_update_if_equals(table, args, key_name, key_value, else_null=False):
    """
    Создает запрос на вставку в таблицу

    Входные параметры
        key_name
            Имя столбца, значение которого нужно проверить

        key_value
            Контрольное значение

        args
            Имена столбцов, которые нужно обновить
            ВНИМАНИЕ: если присутствует столбец key_name, то он должен быть последним в списке args,
            иначе после него проверяться будет новое значение (key_value).

    При наличии такого элемента в таблице, будет произведена попытка обновления
    значения.

    Обновление произойдет, только если значение в столбце key_name равно
    key_value. В противном случае запись останется полностью неизменной
    (а при else_null=True запрос попытается записать в таблицу NULL и скорее всего выбросит ошибку).

    Вызывающая сторона несет ответственность за то, чтобы тип передаваемых
    данных совпадал с типом столбца.
    """
    if key_name in args and args[-1] != key_name:
        raise RuntimeError('INSERT ON DUPLICATE KEY UPDATE: inconsistent parameters')
    return table.insert(
        mysql_on_duplicate_update_keys=args,
        mysql_if_equals=(key_name, key_value),
        mysql_else_null=else_null,
    )


def insert_with_on_duplicate_key_append(table, keys):
    return table.insert(mysql_on_duplicate_append_keys=sorted(keys))


def insert_with_on_duplicate_key_increment(table, key):
    return table.insert(mysql_on_duplicate_increment_key=key)


def is_on_duplicate_key_update_insert(insert):
    return insert.kwargs.get('mysql_on_duplicate_update_keys')


def is_on_duplicate_key_append_insert(insert):
    return insert.kwargs.get('mysql_on_duplicate_append_keys')


def delete_with_limit(table, limit):
    return table.delete(mysql_delete_limit=limit)
