# -*- coding: utf-8 -*-
import bisect
from collections import defaultdict
import logging


log = logging.getLogger('passport.dbmanager')


def build_mod_shard_function(buckets_count):
    def mod_shard_function(key):
        return key % buckets_count
    return mod_shard_function


def build_range_shard_function(ranges):
    """
    ranges: [(shard, begin_interval), ...]
    """
    def range_shard_function(key):
        try:
            position = bisect.bisect(beginning_intervals, key)
        except TypeError:
            raise ValueError('Invalid key value: %r' % key)
        if position == 0:
            raise ValueError('Invalid key value: %r' % key)
        return shards[position - 1]

    ranges = sorted(ranges, key=lambda x: x[1])
    shards = [x[0] for x in ranges]
    beginning_intervals = [x[1] for x in ranges]
    return range_shard_function


class _Sharder(object):
    def __init__(self):
        self.configured = False

    def configure(self, buckets_to_db_names, shard_function=None):
        self.shard_function = shard_function or build_mod_shard_function(len(buckets_to_db_names))
        self.buckets_to_db_names = buckets_to_db_names
        self.configured = True

    def key_to_db_name(self, key):
        bucket_number = self.shard_function(key)
        return self.buckets_to_db_names[bucket_number]

    def db_names(self):
        return list(self.buckets_to_db_names.values())


_sharders = defaultdict(_Sharder)


def get_sharder(table_name):
    return _sharders[table_name]


def get_db_name(table_name, key):
    return get_sharder(table_name).key_to_db_name(key)
