# -*- coding: utf-8 -*-
from collections import OrderedDict
from functools import update_wrapper
import logging
import sys

from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.logging_utils.loggers import GraphiteLogger
import six


log = logging.getLogger('passport.dbmanager.transaction_manager')


class TransactionAdapter(object):
    """
    Вспомогательная прослойка между концепциями SQLAlchemy. Позволяет начать транзакцию в БД,
    выполнить COMMIT или ROLLBACK, обеспечив при этом правильную работу с соединением.
    """
    connection = None
    transaction = None

    def begin(self, engine):
        log.debug('Query: BEGIN')
        conn = engine.connect()
        try:
            transaction = conn.begin()
        except Exception:
            try:
                conn.close()
            finally:
                raise
        conn.reconnect_retries = engine.reconnect_retries
        conn.has_low_timeout = engine.has_low_timeout
        conn.url = engine.url
        self.connection = conn
        self.transaction = transaction

    def commit_and_close(self):
        if self.is_started:
            log.debug('Query: COMMIT')
            self.transaction.commit()
            # В случае ошибки commit - вызывающий вызовет rollback_and_close, где соединение будет закрыто
            self.connection.close()
            self.connection = None

    def rollback_and_close(self):
        if self.is_started:
            try:
                log.debug('Query: ROLLBACK')
                self.transaction.rollback()
            finally:
                self.connection.close()
                self.connection = None

    @property
    def is_started(self):
        return self.connection is not None


@lazy_loadable()
class TransactionManager(object):

    def __init__(self, graphite_logger=None):
        self._reset_state()
        self.graphite_logger = graphite_logger or GraphiteLogger(service='db')

    def _describe(self, engine):
        return engine.url.database

    def _reset_state(self):
        self._open_transactions = OrderedDict()
        self.started = False

    def start_full_transaction(self):
        log.debug('Full transaction started.')
        self._open_transactions = OrderedDict()
        self.started = True

    def rollback_all_transactions(self):
        from passport.backend.core.dbmanager.manager import safe_execute, TRX_STATE_ROLLBACK
        exc = None
        for engine, transaction in six.iteritems(self._open_transactions):
            log.debug('Rolling back transaction on "%s".', self._describe(engine))
            try:
                safe_execute(
                    transaction.connection.engine,
                    lambda engine_: transaction.rollback_and_close(),
                    graphite_logger=self.graphite_logger,
                    retries=1,
                    trx_state=TRX_STATE_ROLLBACK,
                )
            except Exception as e:
                # Отложим исключение, необходимо сделать rollback для всех транзакций
                exc = e
                log.error('Exception while rolling back transaction on "%s"', self._describe(engine), exc_info=exc)
        self._reset_state()
        if exc:
            raise exc

    def commit_all_transactions(self):
        from passport.backend.core.dbmanager.manager import safe_execute, TRX_STATE_COMMIT
        for engine, transaction in six.iteritems(self._open_transactions):
            log.debug('Commiting transaction on "%s".', self._describe(engine))
            safe_execute(
                transaction.connection.engine,
                lambda engine_: transaction.commit_and_close(),
                graphite_logger=self.graphite_logger,
                retries=1,
                trx_state=TRX_STATE_COMMIT,
            )
        self._reset_state()

    def enter_transaction(self, engine):
        if self.started:
            return self._open_transactions.setdefault(engine, TransactionAdapter())
        return TransactionAdapter()


class full_transaction(object):
    """Может использоваться как декоратор или контекстный менеджер"""
    def __init__(self, func=None):
        self.func = func
        self.tm = None
        if func:
            update_wrapper(self, func)

    def __call__(self, *args, **kwargs):
        self.__enter__()
        try:
            result = self.func(*args, **kwargs)
            self.tm.commit_all_transactions()
            return result
        except Exception:
            # Сохраняем exc_info внешнего контекста, чтобы выкинуть
            # исходный стек и исключение
            outer_exc_info = sys.exc_info()
            try:
                self.tm.rollback_all_transactions()
            finally:
                # Бросаем исходное исключение, в нем может быть больше информации, возможные ошибки
                # БД в лог уже записали
                six.reraise(*outer_exc_info)

    def __enter__(self):
        self.tm = get_transaction_manager()
        self.tm.start_full_transaction()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.tm.commit_all_transactions()
        else:
            try:
                self.tm.rollback_all_transactions()
            finally:
                pass  # __exit__ сам перевыкинет исходное исключение, в нем может быть больше информации


def get_transaction_manager():
    tx_manager = LazyLoader.get_instance('TransactionManager')
    return tx_manager
