# -*- coding: utf-8 -*-

from functools import partial

from passport.backend.core.builders.blackbox.blackbox import get_blackbox
from passport.backend.core.builders.blackbox.constants import BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS
from passport.backend.core.dbmanager.manager import safe_execute_queries
from passport.backend.core.models.device_public_key import DevicePublicKey
from passport.backend.core.serializers.eav.query import (
    DeleteDevicePublicKeyQuery,
    InsertDevicePublicKeyQuery,
    UpdateDevicePublicKeyQuery,
)


_blackbox_public_key_found_status_set = {
    BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.OK,
    BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.PUBLIC_KEY_INVALID,
    BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.PUBLIC_KEY_UNSUPPORTED_VERSION,
}


def find_device_public_key(device_id, blackbox=None):
    if not blackbox:
        blackbox = get_blackbox()
    response = blackbox.get_device_public_key(device_id)
    if response['status'] == BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.PUBLIC_KEY_NOT_FOUND:
        return
    if response['status'] in _blackbox_public_key_found_status_set:
        device_public_key = DevicePublicKey()
        device_public_key.parse(
            dict(
                device_id=device_id,
                public_key=response['value'],
                version=int(response['version']),
                owner_id=int(response['owner_id']),
            ),
        )
        return device_public_key
    raise NotImplementedError('Unknown blackbox.get_device_public_key status: %s' % response['status'])


def insert_device_public_key(device_public_key):
    insert_query = InsertDevicePublicKeyQuery(
        device_public_key.device_id,
        device_public_key.owner_id,
        device_public_key.public_key,
        device_public_key.version,
    )
    safe_execute_queries([insert_query])


def update_device_public_key(device_id, owner_id, public_key, version):
    update_query = UpdateDevicePublicKeyQuery(device_id, owner_id, public_key, version)
    public_key_found = [False]
    process_update_result = partial(_process_update_result, public_key_found)
    safe_execute_queries([(update_query, process_update_result)])
    return public_key_found[0]


def delete_device_public_key(device_id, owner_id):
    delete_query = DeleteDevicePublicKeyQuery(device_id, owner_id)
    public_key_found = [False]
    process_update_result = partial(_process_update_result, public_key_found)
    safe_execute_queries([(delete_query, process_update_result)])
    return public_key_found[0]


def _process_update_result(public_key_found, result):
    if result.rowcount == 0:
        public_key_found[0] = False
    elif result.rowcount == 1:
        public_key_found[0] = True
    else:
        message = 'How to deal with %s updated device public keys?' % result.rowcount  # pragma: no cover
        raise RuntimeError(message)  # pragma: no cover
