# -*- coding: utf-8 -*-
from passport.backend.core.differ.types import Diff
from passport.backend.core.differ.utils import (
    get_obj_id,
    model_as_dict,
    normalize_value,
)
from passport.backend.core.models.base import Model
from passport.backend.core.undefined import Undefined
from six import iteritems


def diff(old, new):
    """Возвращает отличия объектов"""
    if new is None:
        return Diff({}, {}, convert(old))
    if old is None:
        return Diff(convert(new), {}, {})
    if not isinstance(old, (Model, dict)) or \
       not isinstance(new, (Model, dict)):
        raise TypeError("diff() arguments should be in (Model, dict)")

    added, changed, deleted = tidy_diff(diff_proxy(old, new))
    return Diff(added or {}, changed or {}, deleted or {})


def convert(obj):
    if isinstance(obj, dict):
        if '_id' in obj:
            del obj['_id']
        new_obj = {}
        for k, v in iteritems(obj):
            if isinstance(v, Model) and v.is_empty():
                continue
            if v != Undefined:
                new_obj[k] = convert(v)
        return new_obj
    elif isinstance(obj, (list, set)):
        if not obj:
            return obj
        old_obj = list(obj)
        sample_value = old_obj[0]
        if type(sample_value) is not tuple:
            new_obj = [(Undefined, convert(x)) for x in old_obj]
        else:
            new_obj = [tuple(map(convert, xs)) for xs in old_obj]
        return type(obj)(new_obj)
    elif isinstance(obj, Model):
        model = model_as_dict(obj)
        return convert(model)
    return obj


def tidy_diff(args):
    """Причёсываем diff"""
    added, changed, deleted = args
    return convert(added), convert(changed), convert(deleted)


def diff_simple(old, new):
    """Для сравнения атомарных типов (int, str)."""
    if old == new:
        return Diff({}, {}, {})
    if old is Undefined:
        return Diff(new, {}, {})
    return Diff({}, new, {})


def diff_proxy(old, new):
    """Вызывает функцию сравнения, соответствующую типам аргументов."""
    if isinstance(old, Model) or isinstance(new, Model):
        return diff_models(old, new)
    elif isinstance(old, dict) or isinstance(new, dict):
        return diff_dicts(old, new)
    elif isinstance(old, list) or isinstance(new, list):
        # Да, списки сравниваются как сеты
        # Я не забочусь о порядке, повторяемости эл-тов и т.п. :-|
        # kmerenkov@ 12.03.2012
        return diff_sets(old or [], new or [])
    return diff_simple(old, new)


def diff_models(old, new):
    """Сравниваем модели"""
    old_model = model_as_dict(old)
    new_model = model_as_dict(new)

    added, changed, deleted = diff_dicts(old_model, new_model)

    return Diff(added, changed, deleted)


def diff_dicts(old, new):
    """Сравниваем словари"""
    if old is None or old is Undefined:
        old = {}
    if new is None or new is Undefined:
        new = {}

    new_set = set(new)
    old_set = set(old)

    new_fields = new_set - old_set
    deleted_fields = old_set - new_set
    common_fields = new_set & old_set

    added = {}
    for key in new_fields:
        new_val = new[key]
        if new_val is not Undefined:
            added[key] = normalize_value(new_val)
    deleted = {}
    for key in deleted_fields:
        deleted[key] = None
    changed = {}

    for key in common_fields:
        # Тут нечего дифать даже, сразу ясно всё
        if (old[key] is not None) and (new[key] is None):
            deleted[key] = None
            continue

        added_val, changed_val, deleted_val = diff_proxy(old[key], new[key])
        if added_val != {}:
            added[key] = added_val
        if changed_val != {}:
            changed[key] = changed_val
        if deleted_val != {}:
            deleted[key] = deleted_val

    return Diff(added or {}, changed or {}, deleted or {})


def diff_sets(old, new):
    """Сравниваем множества"""
    old_map = dict((get_obj_id(f), f) for f in old)
    new_map = dict((get_obj_id(f), f) for f in new)

    added, changed, deleted = diff_dicts(old_map, new_map)
    if added != {}:
        added = [
            (Undefined, normalize_value(v))
            for (_k, v) in iteritems(added)
        ]
    if changed != {}:
        changed = [
            (normalize_value(old_map[k]), normalize_value(v))
            for (k, v) in iteritems(changed)
        ]
    if deleted != {}:
        deleted = [
            (normalize_value(old_map[k]), None)
            for (k, v) in iteritems(deleted)
        ]
    return Diff(added, changed, deleted)
