# -*- coding: utf-8 -*-
import json
import logging
import os
import tempfile

from passport.backend.core.exceptions import BaseCoreError
import six


log = logging.getLogger('passport.disk_cache')


class DiskCacheError(BaseCoreError):
    pass


class DiskCacheReadError(DiskCacheError):
    pass


class DiskCacheWriteError(DiskCacheError):
    pass


class DiskCache(object):
    def __init__(self, filename, folder='/tmp'):
        self._filename = '%s_%d' % (filename, os.getuid())
        self._folder = folder
        self._filepath = os.path.join(folder, self._filename)

    def dump(self, data):
        temp_path = ''
        try:
            _, temp_path = tempfile.mkstemp(prefix=self._filename, dir=self._folder)
            json.dump(data, open(temp_path, 'w'))
            os.rename(temp_path, self._filepath)
        except TypeError as e:
            log.error(u'Failed to convert data to json: %s', six.text_type(e))
            raise DiskCacheWriteError(six.text_type(e))
        except OSError as e:
            log.error(u'Failed to write to file: %s', six.text_type(e))
            raise DiskCacheWriteError(six.text_type(e))
        finally:
            if os.path.isfile(temp_path):
                os.remove(temp_path)

    def load(self):
        try:
            f = open(self._filepath, 'r')
            return json.load(f)
        except IOError as e:
            log.error(u'Failed to read file %s: %s', self._filepath, six.text_type(e))
            raise DiskCacheReadError(six.text_type(e))
        except ValueError as e:
            log.error(u'Failed to decode JSON from %s: %s', self._filepath, six.text_type(e))
            raise DiskCacheReadError(six.text_type(e))

    def flush(self):
        if self.exists():
            os.remove(self._filepath)

    def exists(self):
        return os.path.isfile(self._filepath)


__all__ = (
    'DiskCache',
    'DiskCacheReadError',
    'DiskCacheWriteError',
)
