# -*- coding: utf-8 -*-
from imp import load_source
import json
import logging
from os import path
import random
import time

from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.lazy_loader import LazyLoader
import yaml


log = logging.getLogger('passport.configs')


class LoadConfigsError(BaseCoreError):
    """
    Ошибка загрузки текстовых конфигов
    """


def load_json(filename):
    with open(filename) as f:
        return json.load(f)


def load_yaml(filename):
    with open(filename) as f:
        return yaml.safe_load(f)


def load_plaintext(filename):
    with open(filename) as f:
        return f.read()


class BaseDynamicConfig(object):
    lazy_loadable_requirements = None

    def __init__(self, config_filenames, cache_time):
        self._files_mtimes = {}
        self._cache_time = cache_time
        self.config = None
        self.expires_at = 0
        self.config_filenames = config_filenames or []

    def __str__(self):
        return '%s: expires_at=%s, config=%s' % (self.__class__.__name__, self.expires_at, self.config)

    @property
    def is_expired(self):
        return bool(not self.config or time.time() > self.expires_at)

    def random_cache_time(self):
        return self._cache_time + random.randint(0, self._cache_time / 2)

    def _read_configs(self):
        real_mtimes = self._get_real_mtimes()
        if self._saved_mtimes_match(real_mtimes):
            return

        configs = [self.read_config_file(filepath) for filepath in self.config_filenames]
        if len(configs) == 1:
            merged_configs = configs[0]
        else:
            merged_configs = self.merge_configs(configs)

        self._set_saved_mtimes(real_mtimes)

        return merged_configs

    def _saved_mtimes_match(self, mtimes):
        return bool(self._files_mtimes and self._files_mtimes == mtimes)

    def _set_saved_mtimes(self, mtimes):
        self._files_mtimes = mtimes

    def _get_real_mtimes(self):
        try:
            mtimes = {filepath: int(path.getmtime(filepath)) for filepath in self.config_filenames}
        except OSError:
            mtimes = None
        return mtimes

    def _get_requirements(self):
        """
        Возвращает инстансы конфигов, от которых зависит данный конфиг.
        """
        return (
            LazyLoader.get_instance(requirement_name)
            for requirement_name in (self.lazy_loadable_requirements or [])
        )

    def read_config_file(self, filename):
        """raises: LoadConfigsError"""
        raise NotImplementedError()  # pragma: no cover

    def merge_configs(self, configs):
        raise NotImplementedError()  # pragma: no cover

    def postprocess(self):
        """
        Вызывается после успешного чтения файла.
        Может быть перегружена, если требуется постпроцессинг прочитанных данных.
        """
        pass  # pragma: no cover

    def load(self, force=False):
        if not self.is_expired and not force:
            return

        # Перечитываем все конфиги, от которых зависит текущий
        for requirement in self._get_requirements():
            requirement.load(force=True)

        try:
            new_config = self._read_configs()
        except LoadConfigsError as e:
            if not self.config:
                log.error('Could not load initial %s from file with error: %s', self.__class__.__name__, e)
                raise

            log.warning('Using old %s, could not load from file with error: %s', self.__class__.__name__, e)
            return

        self.expires_at = time.time() + self.random_cache_time()

        # Изменений в конфигах не было, поэтому нам нечего делать
        if new_config is None:
            return

        log.debug('Loaded %s data from file' % self.__class__.__name__)

        old_config = self.config
        self.config = new_config
        try:
            self.postprocess()
        except:
            # Не удалось провести постпроцессинг прочитанных данных - откатываемся на старые данные
            self.config = old_config
            raise

        log.info('Using new %s' % self.__class__.__name__)


class BaseDynamicPyModule(BaseDynamicConfig):
    def __init__(self, py_filename, import_as_name, cache_time=60):
        super(BaseDynamicPyModule, self).__init__(
            config_filenames=[py_filename],
            cache_time=cache_time,
        )
        self.import_as_name = import_as_name

    def read_config_file(self, filename):
        try:
            return load_source(self.import_as_name, filename)
        except Exception as e:
            raise LoadConfigsError(e)


__all__ = (
    'load_json',
    'load_plaintext',
    'load_yaml',
    'BaseDynamicConfig',
    'BaseDynamicPyModule',
    'LoadConfigsError',
)
