# -*- coding: utf-8 -*-
import json

from passport.backend.core.builders.blackbox import (
    BLACKBOX_CHECK_SIGN_STATUS_OK,
    get_blackbox,
)
from passport.backend.core.exceptions import BaseCoreError


class EncryptedContainerBaseError(BaseCoreError):
    """Базовая ошибка работы с контейнером"""


class EncryptedContainerUnknownType(EncryptedContainerBaseError):
    """Неизвестный тип контейнера"""


class EncryptedContainerInvalid(EncryptedContainerBaseError):
    """Что-то не так с контейнером:
    * либо перепутано окружение и не подошел ключ,
    * либо подпись не сошлась.
    """


class BaseEncryptedContainer(object):
    """
    Хранит зашифрованные и подписанные ЧЯ произвольные данные.
    """
    type_to_ttl = {}
    type_to_sign_space = {}
    default_type = None

    def __init__(self, data, container_type=None):
        self.data = data
        self.container_type = container_type or self.default_type
        if self.container_type not in self.type_to_ttl:
            raise EncryptedContainerUnknownType()
        self.ttl = self.type_to_ttl[self.container_type]

    @classmethod
    def _get_sign_space(cls, container_type):
        return cls.type_to_sign_space.get(container_type, container_type)

    def __getitem__(self, item):
        return self.data[item]

    def __setitem__(self, key, value):
        self.data[key] = value

    def pack(self):
        to_sign = json.dumps(self.data)
        signed_data = get_blackbox().sign(
            value=to_sign,
            ttl=self.ttl,
            sign_space=self._get_sign_space(self.container_type),
        )
        return signed_data['signed_value']

    @classmethod
    def unpack(cls, packed_container, container_type=None):
        container_type = container_type or cls.default_type
        signed_data = get_blackbox().check_sign(packed_container, sign_space=cls._get_sign_space(container_type))

        if signed_data['status'] != BLACKBOX_CHECK_SIGN_STATUS_OK:
            raise EncryptedContainerInvalid(signed_data['status'])

        data = json.loads(signed_data['value'])

        return cls(data, container_type)
