# -*- coding: utf-8 -*-
from collections import namedtuple
import logging

from passport.backend.core.conf import settings
from passport.backend.core.env_profile.helpers import make_profile_from_raw_data
from passport.backend.core.portallib import get_uatraits
from passport.backend.core.types.ip.ip import IP
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_str
from six import iteritems


log = logging.getLogger('passport.env')


def parse_uatraits_value(value):
    if not value or value == 'Unknown':
        return
    return value


class Environment(
    namedtuple(
        'EnvBase',
        'cookies cookies_all consumer_ip user_ip user_agent host '
        'accept_language referer authorization service_ticket '
        'user_ticket request_id request_path'
    )
):
    def __new__(cls, **kwargs):
        result = dict((i, None) for i in cls._fields)
        result['cookies'] = {}
        result['cookies_all'] = {}
        result.update(kwargs)
        result['consumer_ip'] = IP(result['consumer_ip'])

        if result['user_ip'] is not None:
            result['user_ip'] = IP(result['user_ip'])
        return super(Environment, cls).__new__(cls, **result)

    @property
    def session_id(self):
        return self.cookies.get('Session_id')

    @property
    def ssl_session_id(self):
        return self.cookies.get('sessionid2')

    @property
    def sessguard(self):
        return self.cookies.get('sessguard')

    @cached_property
    def user_agent_info(self):
        raw_user_agent_info = get_uatraits().detect_by_headers({
            'User-Agent': smart_str(self.user_agent or ''),
            'Accept-Language': smart_str(self.accept_language or ''),
        })
        user_agent_info = {}
        for k, v in iteritems(raw_user_agent_info):
            cleaned_value = parse_uatraits_value(v)
            if cleaned_value is not None:
                user_agent_info[k] = cleaned_value
        return user_agent_info

    def make_profile(self, is_mobile=None, mobile_params=None):
        browser_name = self.user_agent_info.get('BrowserName')
        os_name = self.user_agent_info.get('OSName', self.user_agent_info.get('OSFamily'))
        os_version = self.user_agent_info.get('OSVersion')  # тут могут прийти буквы, судя по доке?
        yandexuid = self.cookies.get('yandexuid')

        additional_params = {}
        if is_mobile:
            additional_params = mobile_params or {}
        else:
            # сценарий для веба
            if browser_name not in settings.BROWSER_ENCODE:
                log.debug('make_profile - unknown browser: "%s"', browser_name)
            if (os_name, os_version) not in settings.OS_ENCODE and os_name not in settings.OS_DUMB_ENCODE:
                log.debug('make_profile - unknown os: "%s" or version: "%s"', os_name, os_version)

        return make_profile_from_raw_data(self.user_ip, yandexuid, self.user_agent_info, is_mobile=is_mobile,
                                          **additional_params)
