# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import datetime
import logging
import uuid

from passport.backend.core.builders.ufo_api import (
    BaseUfoApiError,
    get_ufo_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.env_profile.profiles import UfoProfile
from passport.backend.core.ydb import get_ydb_profile
from passport.backend.core.ydb.exceptions import BaseYdbError
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger('passport.env_profile.loader')

# Следующие две функции необходимы для преобразования данных из YDB
# в формат, полностью соответствоваший тому, что хранилось в Кассандре.


# Ported from uuid.py
def uuid1(nanoseconds, unique_id):
    # 0x01b21dd213814000 is the number of 100-ns intervals between the
    # UUID epoch 1582-10-15 00:00:00 and the Unix epoch 1970-01-01 00:00:00.
    timestamp = int(nanoseconds / 100) + 0x01b21dd213814000
    time_low = timestamp & 0xffffffff
    time_mid = (timestamp >> 32) & 0xffff
    time_hi_version = (timestamp >> 48) & 0x0fff
    clock_seq_low = unique_id & 0xff
    clock_seq_hi_variant = (unique_id >> 8) & 0x3f
    node_id = unique_id >> 14
    return uuid.UUID(
        fields=(
            time_low, time_mid,
            time_hi_version, clock_seq_hi_variant,
            clock_seq_low, node_id,
        ),
        version=1,
    )


def repack_ydb_to_ufo(items, glogout_timestamp=None):
    new_items = []
    for item in items:
        item = dict(item)
        wrapped = {}
        if glogout_timestamp and 'timestamp' in item and item['timestamp'] < glogout_timestamp:
            continue
        if item.get('profile_type', '') == 'full':
            wrapped['id'] = UfoProfile.PROFILE_FAKE_UUID
        else:
            timestamp = item.pop('timestamp', 0) * 10 ** 6 or 2 ** 64 - 1
            wrapped['id'] = str(uuid1(timestamp * 1000, unique_id=0))
        wrapped['data'] = item
        new_items.append(wrapped)
    return new_items


ProfileResult = namedtuple('ProfileResult', ['source', 'success', 'content'])


def load_ufo_profile(uid, from_datetime=None, safe=True, statbox=None, statbox_context=None, force_ydb=False):
    ufo_profile_items = []
    ydb_profile_items = []
    status_ufo = False
    status_ydb = False
    log_record = {}
    glogout_timestamp = datetime_to_integer_unixtime(from_datetime) if isinstance(from_datetime, datetime) else None

    is_ydb_used = force_ydb or (uid % 100) < settings.YDB_PERCENTAGE

    if is_ydb_used:
        if statbox_context:
            log_record.update(statbox_context)

        try:
            ydb_profile_items = list(get_ydb_profile().get_profile(uid))
            log_record['ydb_profile_items'] = str(ydb_profile_items)
            status_ydb = True
        except BaseYdbError as e:
            log.warning('Failed to load profile from YDB: %s: %s', type(e), e)
            log_record.update({'status': 'error', 'error': str(e)})

        if (settings.TRY_USE_YDB and status_ydb) or force_ydb:
            if statbox:
                statbox.log(**log_record)
            return ProfileResult(
                'ydb',
                status_ydb,
                UfoProfile(
                    repack_ydb_to_ufo(ydb_profile_items, glogout_timestamp=glogout_timestamp),
                    data_decoder=lambda x: x,
                ),
            )

    try:
        ufo_api = get_ufo_api()
        ufo_profile_items = ufo_api.profile(uid)
        log_record['ufo_profile_items'] = str(ufo_profile_items)
        status_ufo = True
    except BaseUfoApiError as e:
        log.warning('Failed to load profile from UFO: %s: %s', type(e), e)
        if not safe:
            raise

    if statbox and is_ydb_used:
        statbox.log(**log_record)

    return ProfileResult('ufo', status_ufo, UfoProfile(ufo_profile_items))
