# -*- coding: utf-8 -*-
from collections import namedtuple


EnvDistance = namedtuple('EnvDistance', 'Min Low Moderate Medium High Max')._make([0, 0, 4, 10, 100, 100])


class BaseEnvMetric(object):
    @staticmethod
    def distance(new, old):
        raise NotImplementedError()  # pragma: no cover


class Basic3x3CaseMetric(BaseEnvMetric):
    @staticmethod
    def distance(new, old):
        if new.is_mobile or old.is_mobile:
            return Basic3x3CaseMetric.distance_mobile(new, old)
        else:
            return Basic3x3CaseMetric.distance_web(new, old)

    @staticmethod
    def distance_mobile(new, old):
        # Дистанция считается небольшой, только если АСки совпали.
        # Если их нет в профиле или они не совпали, то дистанция уже считается большой.
        # Сейчас делаем так, что логин из новой сети каждый раз => челлендж.
        if new.AS_list and new.AS_list == old.AS_list:
            geo_distance = EnvDistance.Low
        else:
            geo_distance = EnvDistance.High
        return geo_distance

    @staticmethod
    def distance_web(new, old):
        # geo
        # АСки совпадают
        if new.AS_list and new.AS_list == old.AS_list:
            geo_distance = EnvDistance.Low
        elif (
            # АСок по какой-то причине нет, но есть страна и город, и они совпали
            not new.AS_list and
            new.country_id is not None and new.country_id == old.country_id and
            new.city_id == old.city_id
        ):
            geo_distance = EnvDistance.Low
        # АСки есть, но не совпали, тогда сравниваются страны
        elif new.country_id is not None and new.country_id == old.country_id:
            geo_distance = EnvDistance.Medium
        # если страны не совпали, то
        else:
            geo_distance = EnvDistance.High

        # user_agent
        if (
            new.browser_id is not None and new.browser_id == old.browser_id and
            new.os_id is not None and new.os_id == old.os_id and
            new.yandexuid_timestamp is not None
        ):
            if new.yandexuid_timestamp == old.yandexuid_timestamp:
                user_agent_distance = EnvDistance.Low
            elif new.yandexuid_timestamp > old.yandexuid_timestamp:
                user_agent_distance = EnvDistance.Moderate
            else:
                user_agent_distance = EnvDistance.High
        elif new.is_mobile or old.is_mobile:
            user_agent_distance = EnvDistance.Moderate
        else:
            user_agent_distance = EnvDistance.High

        return (geo_distance + user_agent_distance) / 2
