# -*- coding: utf-8 -*-
import json
from time import time
import uuid

from passport.backend.core.utils.decorators import cached_property


class BaseEnvProfile(object):
    version = None
    fields = None

    def __init__(self, timestamp=None, timeuuid=None, raw_env=None, **kwargs):
        unknown_fields = [field for field in kwargs if field not in self.fields]
        if unknown_fields:
            raise ValueError('Unknown fields: %s' % unknown_fields)

        # общие поля
        self.timestamp = timestamp or int(time())
        self.timeuuid = timeuuid or uuid.uuid1().bytes
        self.raw_env = raw_env
        # параметры профиля
        for field in self.fields:
            setattr(self, field, kwargs.get(field))

    def __eq__(self, other):
        if type(self) != type(other):
            return False
        for field in self.all_fields:
            if getattr(self, field) != getattr(other, field):
                return False
        return True

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        return '<%s ts=%d uuid=%s>' % (self.__class__.__name__, self.timestamp, uuid.UUID(bytes=self.timeuuid))

    @property
    def age(self):
        return int(time() - self.timestamp)

    @property
    def all_fields(self):
        return self.fields + ['timestamp', 'timeuuid']

    @cached_property
    def as_dict(self):
        return dict(
            {field: getattr(self, field) for field in self.all_fields if field != 'timeuuid'},
            timeuuid=str(uuid.UUID(bytes=self.timeuuid)),
        )

    @cached_property
    def as_json(self):
        return json.dumps(self.as_dict)


class EnvProfileV1(BaseEnvProfile):
    version = 1
    fields = [
        'AS_list',
        'country_id',
        'city_id',
        'browser_id',
        'is_mobile',
        'os_id',
        'yandexuid_timestamp',
        'am_version',
        'device_id',
        'cloud_token',
    ]
