# -*- coding: utf-8 -*-
import base64
import json
import logging
import uuid

from passport.backend.core.env_profile.helpers import make_profile_from_raw_data
from passport.backend.core.env_profile.metric import (
    Basic3x3CaseMetric,
    EnvDistance,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import timeuuid_to_timestamp


log = logging.getLogger('passport.env_profile.profiles')


class ProcessNewProfileResult(object):
    WRITE_TO_STORAGE = False

    def __init__(self, closest_distance):
        self.closest_distance = closest_distance


class ProcessNewProfileLimitedResult(ProcessNewProfileResult):
    pass


class ProcessNewProfileUpdatedResult(ProcessNewProfileResult):
    WRITE_TO_STORAGE = True

    def __init__(self, profile, *args, **kwargs):
        super(ProcessNewProfileUpdatedResult, self).__init__(*args, **kwargs)
        self.profile = profile


class UfoProfile(object):

    # Специальный TimeUUID для записей полного профиля.
    PROFILE_FAKE_UUID = 'ffffffff-ffff-1fff-bfff-ffffffffffff'

    def __init__(self, profile_items, data_decoder=None):
        self._full_profile = None
        fresh_items = []
        for profile_item in profile_items:
            # первый найденный профиль с id полного профиля будет полным, все что нашли дальше - fresh профилями
            if not self._full_profile and profile_item.get('id') == self.PROFILE_FAKE_UUID:
                self._full_profile = profile_item['data']
            else:
                fresh_items.append(profile_item)
        self._fresh_items_iter = iter(fresh_items)
        self._fresh_profiles_cache = []
        self._data_decoder = data_decoder
        if self._data_decoder is None:
            self._data_decoder = lambda x: json.loads(base64.b64decode(x))

    @property
    def fresh_profiles(self):
        """
        Возвращает генератор обработанных fresh-профилей. При повторных вызовах
        уже обработанные профили берутся из кеша. Нельзя одновременно работать с двумя генераторами.
        """
        for env_profile in self._fresh_profiles_cache:
            yield env_profile

        for fresh_item in self._fresh_items_iter:
            try:
                data = self._data_decoder(fresh_item['data'])
                timeuuid = uuid.UUID(hex=fresh_item['id'])
                env_profile = make_profile_from_raw_data(
                    timeuuid=timeuuid.bytes,
                    timestamp=timeuuid_to_timestamp(timeuuid),
                    **data
                )
            except (TypeError, ValueError) as e:
                log.warning('Failed to parse fresh data from UFO: %s: %s (item %s)', type(e), e, fresh_item)
                continue

            self._fresh_profiles_cache.append(env_profile)
            yield env_profile

    def find_closest(self, new_profile, metric=Basic3x3CaseMetric, default_distance=EnvDistance.Max):
        profiles_distances = []
        for fresh_profile in self.fresh_profiles:
            distance = metric.distance(new=new_profile, old=fresh_profile)
            profiles_distances.append((fresh_profile, distance))
            if distance == EnvDistance.Min:
                # Нашли идентичный профиль - выходим, чтобы не парсить лишние данные
                return fresh_profile, distance

        if profiles_distances:
            return min(
                profiles_distances,
                key=lambda k: k[1],
            )
        return None, default_distance

    def process_new_profile(self, new_profile, metric=Basic3x3CaseMetric):
        closest_profile, distance = self.find_closest(new_profile, metric=metric)
        if distance <= EnvDistance.Moderate:
            # Заменяем timeuuid на существующий для того, чтобы обновить fresh-профиль
            new_profile.timeuuid = closest_profile.timeuuid
        return ProcessNewProfileUpdatedResult(
            new_profile,
            closest_distance=distance,
        )

    @cached_property
    def full_profile(self):
        profile = None
        if self._full_profile:
            try:
                profile = self._data_decoder(self._full_profile)
            except (TypeError, ValueError) as e:
                log.warning('Failed to parse full profile from UFO: %s: %s', type(e), e)
        return profile

    @property
    def trusted_device_ids(self):
        result = set()
        if self.full_profile:
            for device_id, _ in self.full_profile.get('it_device_id_freq_6m', []):
                result.add(device_id)
        for fresh_profile in self.fresh_profiles:
            if fresh_profile.device_id:
                result.add(fresh_profile.device_id)
        return result

    @property
    def trusted_cloud_tokens(self):
        result = set()
        if self.full_profile:
            for cloud_token, _ in self.full_profile.get('it_cloud_token_freq_6m', []):
                result.add(cloud_token)
        for fresh_profile in self.fresh_profiles:
            if fresh_profile.cloud_token:
                result.add(fresh_profile.cloud_token)
        return result
