# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import datetime
import logging
import random
import time

from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    Blackbox,
    BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
)
from passport.backend.core.builders.frodo import (
    FrodoError,
    get_frodo,
)
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger('passport.frodo')
exception_log = logging.getLogger('passport.frodo.error')


KARMA_MIN = 75
KARMA_TIME = 1200

FrodoStatus = namedtuple(
    'FrodoStatus',
    'is_karma_prefix_returned is_karma_suffix_returned',
)


def get_account(user):
    data = Blackbox().userinfo(
        login=user.login,
        find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
    )
    account = Account().parse(data)

    return account


def generate_karma(user):
    activation_datetime = karma_prefix = karma_suffix = None

    if user.spam:
        if user.karma >= KARMA_MIN:
            activation_datetime = datetime.fromtimestamp(
                int(time.time()) + KARMA_TIME + random.randint(0, 600),
            )
        karma_prefix = 0
        karma_suffix = user.karma
    elif user.change_pass:
        karma_prefix = 7 if user.karma == 85 else 8

    return karma_prefix, karma_suffix, activation_datetime


def update_karma(account, karma_prefix, karma_suffix, activation_datetime, env,
                 consumer, action='karma'):
    events = {
        'action': action,
        'consumer': consumer,
    }
    with UPDATE(account, env, events):
        if activation_datetime is not None:
            account.karma.activation_datetime = activation_datetime
        if karma_prefix is not None:
            account.karma.prefix = karma_prefix
        if karma_suffix is not None:
            account.karma.suffix = karma_suffix

    return account


def process_user_karma(user, env, consumer, registered_account, action_to_log=None):
    registered_account_login = registered_account.login if registered_account.is_pdd else registered_account.normalized_login

    if registered_account_login != user.login:
        # Этого пользователя ФО прислал "задним числом" - запишем в статбокс-лог это событие
        account = get_account(user)
        action_to_log = 'pending_karma_update'
    else:
        account = registered_account
        action_to_log = action_to_log or 'karma'

    karma_prefix, karma_suffix, activation_datetime = generate_karma(user)

    account = update_karma(
        account,
        karma_prefix,
        karma_suffix,
        activation_datetime,
        env,
        consumer,
        action=action_to_log,
    )

    fs = FrodoStatus(
        is_karma_prefix_returned=karma_prefix is not None,
        is_karma_suffix_returned=karma_suffix is not None,
    )

    return account.uid, fs


def check_spammer(frodo_info, env, consumer, registered_account, frodo=None):
    """
    Нельзя вызывать в контексте CREATE/UPDATE.
    """
    # Словарь со статусами изменения кармы по каждому пользователю
    frodo_statuses = {}

    try:
        if not frodo:
            frodo = get_frodo()

        # Вызываем Фродо
        response = frodo.check(frodo_info)

        # Обрабатываем ответ и проставляем карму
        processed = {}

        for user in response:
            try:
                uid, fs = process_user_karma(
                    user,
                    env,
                    consumer,
                    registered_account,
                    action_to_log=frodo_info.action,
                )

                frodo_statuses[uid] = fs

            except (BaseBlackboxError, DBError) as e:
                log.error('Error during processing karma for %s: %s', user, e)
                # Квитанцию для этого логина в ФО не отправляем - пусть пришлют его ещё раз
                continue
            except UnknownUid:
                # Аккаунт удалили, но это нормально: квитанцию в ФО надо отправить
                log.warning('User %s not found', user)
            except Exception as e:
                # Хотим всегда отправлять ФО квитанцию о доставке.
                # Поэтому ошибку проглатываем, но в statbox и exception-логи пишем
                to_statbox({
                    'event': 'account_modification',
                    'destination': 'frodo',
                    'entity': 'karma',
                    'action': 'unhandled_error',
                    'login': user.login,
                    'error': str(e),
                })
                # тут у нас нет реквеста, поэтому пишем усеченную запись об ошибке
                exception_log.error(
                    u'%s (%s) while processing %s for SO' % (e.__class__.__name__, e.__class__, user),
                    exc_info=e,
                    extra={
                        'request_info': '',
                        'request_headers': {},
                        'request_values': {},
                    },
                )

            processed[user.login] = str(user.karma)
            if user.change_pass:
                processed[user.login] += '-'

        # Пишем в лог если в ответе Фродо нет текущего логина
        login = frodo_info.login
        if login not in processed:
            log.debug('Login "%s" not in SO result for action "%s"', login, frodo_info.action)

        # подтверждаем обработанных пользователей
        if processed:
            frodo.confirm(processed)

    except FrodoError as ex:
        log.error('Error during SO call: %s', ex, exc_info=ex)

    frodo_status = frodo_statuses.get(
        registered_account.uid,
        FrodoStatus(
            is_karma_prefix_returned=False,
            is_karma_suffix_returned=False,
        ),
    )

    return registered_account, frodo_status
