# -*- coding: utf-8 -*-
import bisect
from collections import namedtuple

from geobase6 import Lookup
import netaddr
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)


COUNTRY_REGION_TYPE = 3
CITY_REGION_TYPE = 6
GEOBASE_YANDEX_REGION = 9999


# Отображение домена в имя страны, по которому Геобаза может получить регион
_DOMAIN_TO_COUNTRY_CODES = {
    'ru': 'ru',
    'com': 'us',
    'kz': 'kz',
    'by': 'by',
    'ua': 'ua',
    'com.tr': 'tr',
    'az': 'az',
    'com.am': 'am',
    'com.ge': 'ge',
    'co.il': 'il',
    'kg': 'kg',
    'lt': 'lt',
    'lv': 'lv',
    'md': 'md',
    'tj': 'tj',
    'tm': 'tm',
    'uz': 'uz',
    'fr': 'fr',
    'ee': 'ee',
}


@lazy_loadable()
class Geobase(object):
    def __init__(self, geobase_lookup_file=None):
        self.lookup = Lookup(geobase_lookup_file or settings.GEOBASE_LOOKUP_FILE)
        self.countries = self.get_countries()

    def get_countries(self):
        countries = {}
        for region in self.lookup.get_regions_by_type(COUNTRY_REGION_TYPE):
            countries[region['short_en_name'].lower()] = region
        return countries

    def asset(self, ip, safe=False):
        try:
            return self.lookup.get_as_by_ip(ip).get('as_list')[0]
        except (RuntimeError, ValueError):
            if safe:
                return None
            raise

    def region(self, ip):
        return self.lookup.get_region_by_ip(ip)

    def regions(self, ip):
        return self.lookup.get_regions_path(ip)

    def __getattr__(self, item):
        return getattr(self.lookup, item)


def get_geobase():
    return LazyLoader.get_instance('Geobase')


def get_first_parent_by_condition(region_id, condition, geobase=None):
    geobase = geobase or get_geobase()
    region = geobase.get_region_by_id(region_id)

    if condition(region):
        return region

    for parent_id in geobase.get_parents_ids(region_id):
        region = geobase.get_region_by_id(parent_id)
        if condition(region):
            return region

    return None


def is_valid_country_code(code, geobase=None):
    geobase = geobase or get_geobase()
    code = code.lower()
    return bool(code) and code in geobase.countries


def get_country_code_by_ip(ip, geobase=None, lower=False):
    country = Region(ip=str(ip), geobase=geobase).country
    if country:
        iso_name = country.get('iso_name')
        if iso_name and lower:
            return iso_name.lower()
        return iso_name


class Region(object):
    def __init__(self, ip=None, tld=None, id=None, geobase=None):
        self.geobase = geobase or get_geobase()
        self.ip = str(ip) if ip else ip
        self.tld = tld.lower() if tld else tld
        self.__country = None
        self.__city = None
        self.__region = None

        self.__AS_list = []
        if id is not None:
            self.__region = self.geobase.get_region_by_id(id)

        self._find_region()
        self._find_country()

    def _find_region(self):
        if self.__region is not None:
            return

        if self.ip:
            try:
                self.__region = self.geobase.region(self.ip)
            except (RuntimeError, ValueError):
                pass
        elif self.tld is not None:
            country_code = _DOMAIN_TO_COUNTRY_CODES.get(self.tld, settings.DEFAULT_COUNTRY)
            self.__country = self.__region = self.geobase.countries[country_code]

    def _find_region_by_type(self, region_type):
        if self.__region is not None:
            return get_first_parent_by_condition(self.__region['id'], lambda r: r['type'] == region_type, geobase=self.geobase)

    def _find_country(self):
        if self.__country is not None:
            return
        self.__country = self._find_region_by_type(COUNTRY_REGION_TYPE)

    def _find_city(self):
        if self.__city is not None:
            return
        self.__city = self._find_region_by_type(CITY_REGION_TYPE)

    def _find_AS_list(self):
        if self.__AS_list:
            return
        asname = self.geobase.asset(self.ip, safe=True)
        if asname:
            self.__AS_list = [asname]

    @property
    def country(self):
        return self.__country

    @property
    def city(self):
        self._find_city()
        return self.__city

    @property
    def AS_list(self):
        self._find_AS_list()
        return self.__AS_list

    @property
    def id(self):
        return None if self.__region is None else self.__region['id']

    @property
    def name(self):
        return '' if self.__region is None else self.__region['name']

    @property
    def lat(self):
        return None if self.__region is None else self.__region['latitude']

    @property
    def lon(self):
        return None if self.__region is None else self.__region['longitude']

    @property
    def timezone(self):
        return None if self.__region is None else self.__region.get('tzname')

    def linguistics(self, language):
        return None if self.__region is None else self.geobase.get_linguistics(self.__region['id'], language)


IPLookupResult = namedtuple('IPLookupResult', 'subnet as_list')

ASLookupResult = namedtuple('ASLookupResult', 'aliases descriptions')


@lazy_loadable()
class ASLookup(object):
    def __init__(self, ipv4_origin_file=None, ipv6_origin_file=None, as_name_file=None):
        self._ipv4_origin_file = ipv4_origin_file or settings.GEOBASE_IPV4_ORIGIN_FILE
        self._ipv6_origin_file = ipv6_origin_file or settings.GEOBASE_IPV6_ORIGIN_FILE
        self._as_name_file = as_name_file or settings.GEOBASE_AS_NAME_FILE

        self._generate_lookups()

    def _build_ip_lookup(self, filename):
        interval_properties = {}
        interval_start_list = []
        with open(filename, 'r') as file_obj:
            for line in file_obj:
                splitted = line.strip('\n').split('\t')
                ip0, ip1 = map(int, splitted[0:2])
                as_list = splitted[2:]

                interval_properties[ip0] = dict(
                    interval_end=ip1,
                    as_list=[as_name.lower() for as_name in as_list],
                )
                # Интервалы гарантированно упорядочены и не пересекаются
                interval_start_list.append(ip0)
            return dict(
                interval_start_list=interval_start_list,
                interval_properties=interval_properties,
            )

    def _build_as_name_lookup(self):
        as_lookup = {}
        with open(self._as_name_file, 'rb') as file_obj:
            for line in file_obj:
                as_name, as_alias, as_description = line.decode('utf8', 'replace').strip('\n').split('\t')
                if as_name.lower() not in as_lookup:
                    as_lookup[as_name.lower()] = ASLookupResult([], [])
                as_data = as_lookup[as_name.lower()]
                as_data.aliases.append(as_alias)
                as_data.descriptions.append(as_description)
        return as_lookup

    def _generate_lookups(self):
        self._as_name_lookup = self._build_as_name_lookup()
        self._ipv4_lookup = self._build_ip_lookup(self._ipv4_origin_file)
        self._ipv6_lookup = self._build_ip_lookup(self._ipv6_origin_file)

    def _find_interval_for_ip(self, integer_ip, interval_start_list, interval_properties):
        index = bisect.bisect_right(interval_start_list, integer_ip)
        if index:
            interval_start = interval_start_list[index - 1]
            properties = interval_properties[interval_start]
            if integer_ip <= properties['interval_end']:
                return interval_start, properties
        return None, None

    def ip_lookup(self, ip):
        if ip is None:
            return None
        netaddr_ip = netaddr.IPAddress(ip)
        version = netaddr_ip.version

        lookup = self._ipv4_lookup if version == 4 else self._ipv6_lookup
        integer_ip = int(netaddr_ip)

        interval_start, properties = self._find_interval_for_ip(integer_ip, **lookup)
        if interval_start is not None:
            interval_start = netaddr.IPAddress(interval_start, version=version)
            interval_end = netaddr.IPAddress(properties['interval_end'], version=version)
            subnets = netaddr.IPRange(interval_start, interval_end).cidrs()
            for subnet in subnets:
                if netaddr_ip in subnet:
                    return IPLookupResult(subnet, properties['as_list'])

    def as_name_lookup(self, as_name):
        return self._as_name_lookup.get(as_name.lower())


def get_as_lookup():
    return LazyLoader.get_instance('ASLookup')
