# -*- coding: utf-8 -*-
from passport.backend.core.dynamic_config import LoadConfigsError
from passport.backend.core.exceptions import BaseCoreError
from six import iteritems


class LoadGrantsError(LoadConfigsError):
    """
    Ошибка загрузки потребительских грантов в паспорт
    """


class GrantsError(BaseCoreError):
    def __init__(self, ip, consumer, tvm_client_id):
        self.ip = ip
        self.consumer = consumer
        self.tvm_client_id = tvm_client_id
        super(GrantsError, self).__init__()

    def _to_string(self, **extra_kwargs):
        kwargs = dict(
            ip=self.ip,
            consumer=self.consumer,
            tvm_client_id=self.tvm_client_id,
        )
        kwargs.update(extra_kwargs)
        description = ', '.join(
            '%s=%s' % (key, value)
            for key, value in sorted(iteritems(kwargs))
        )
        return '<%s: %s>' % (self.__class__.__name__, description)

    def __str__(self):
        return self._to_string()


class MissingGrantsError(GrantsError):
    def __init__(self, missing, ip, consumer, tvm_client_id):
        self.missing = missing
        super(MissingGrantsError, self).__init__(ip, consumer, tvm_client_id)

    def __str__(self):
        return self._to_string(missing_grants=self.missing)


class MissingRequiredGrantsError(MissingGrantsError):
    """
    Не хватает обязательных грантов.
    """


class MissingOptionalGrantsError(MissingGrantsError):
    """
    Не хватает дополнительных грантов.
    """


class UnknownConsumerError(MissingGrantsError):
    "Потребителю не разрешается приходить с того адреса с которого он пришёл."


class InvalidSourceError(GrantsError):
    """
    Пользователь пришел с тикетом от неправильного источника
    """


class TicketParsingError(GrantsError):
    """
    Ошибка при парсинге тикета
    """
    def __init__(self, debug_info, message, status, ip, consumer, tvm_client_id):
        self.debug_info = debug_info
        self.message = message
        self.status = status
        super(TicketParsingError, self).__init__(ip, consumer, tvm_client_id)

    def __str__(self):
        return self._to_string(debug_info=self.debug_info, message=self.message)


class MissingTicketError(GrantsError):
    """
    Отсутствующий тикет
    """
