# -*- coding: utf-8 -*-

import logging

from frozendict import frozendict
from passport.backend.core.grants.exceptions import (
    InvalidSourceError,
    MissingOptionalGrantsError,
    MissingRequiredGrantsError,
    MissingTicketError,
    TicketParsingError,
    UnknownConsumerError,
)
from passport.backend.core.grants.grants_config import (
    get_grants_config,
    Permission,
)
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.tvm import get_tvm_credentials_manager
from ticket_parser2.exceptions import TicketParsingException


log = logging.getLogger('passport.grants')


@lazy_loadable()
class Grants(object):
    """
    Класс, реализующий логику проверки грантов и сервисных тикетов
    """

    def __init__(self, graphite_logger=None):
        self.graphite_logger = graphite_logger or GraphiteLogger(service='grants')

    def _invoke_callable(self, required_grants, *args, **kwargs):
        extended_required_grants_set = set()
        for item in required_grants:
            if callable(item):
                extended_required_grants_set.update(item(*args, **kwargs) or [])
            else:
                extended_required_grants_set.add(item)
        return extended_required_grants_set

    def parse_service_ticket(self, service_ticket, ip, consumers):
        if not service_ticket:
            return
        try:
            return get_tvm_credentials_manager().service_context.check(service_ticket)
        except TicketParsingException as ex:
            joined_consumers = ','.join(consumers)
            self.graphite_logger.log(
                status='error',
                reason='invalid_ticket',
                ip=ip,
                consumers=joined_consumers,
                debug_info=ex.debug_info,
                message=ex.message,
            )
            raise TicketParsingError(ex.debug_info, ex.message, ex.status, ip, joined_consumers, tvm_client_id=None)

    def check_access(self, ip, consumers, required_grants,
                     optional_grants=None, grants_args=tuple(), grants_kwargs=frozendict(), service_ticket=None):
        """
        Проверка грантов и сервисного тикета
        :param ip:
        :param consumers:
        :param required_grants: список требуемых грантов
        :param optional_grants: список опциональных грантов
        :param grants_args: параметры для callable грантов
        :param grants_kwargs: параметры для callable грантов
        :param service_ticket: тело сервисного тикета
        :raises TicketParsingException, InvalidSourceError, MissingRequiredGrantsError,
        MissingOptionalGrantsError, UnknownConsumerError
        :return:
        """
        optional_grants = set(optional_grants or [])
        required_grants = self._invoke_callable(required_grants, *grants_args, **grants_kwargs)

        parsed_ticket = self.parse_service_ticket(service_ticket, ip, consumers)
        tvm_client_id = parsed_ticket.src if parsed_ticket is not None else None
        self.graphite_logger.bind_context(
            ip=ip,
            tvm_client_id=tvm_client_id or '-',
        )

        permission = Permission(
            is_allowed=False,
            is_valid_ip=False,
            is_valid_client_id=False,
            missing_required=required_grants,
            missing_optional=optional_grants,
        )
        for consumer in consumers:
            permission = get_grants_config().is_permitted(
                required_grants,
                optional_grants,
                ip,
                consumer,
                tvm_client_id,
            )

            if permission.is_allowed:
                break

        # Мы смотрим только на результат проверки IP по последнему
        # потребителю. Возможны 2 ситуации:
        #   а) потребитель один
        #   б) потребителей много, но для всех is_valid_ip == True,
        #      т.к. все они были получены из программы
        #      get_consumers(ip), когда consumer не был задан явно.
        # Других ситуаций возникнуть не может, т.к. мы перешли на
        # использование явного потребителя.
        if not permission.is_allowed and not permission.is_valid_ip:
            if not consumers and not get_grants_config().get_consumers(ip):
                self.graphite_logger.log(
                    status='error',
                    reason='consumers_not_found',
                )
                raise UnknownConsumerError(
                    list(required_grants),
                    ip,
                    '',
                    tvm_client_id,
                )
            else:
                joined_consumers = ','.join(consumers)
                self.graphite_logger.log(
                    status='error',
                    reason='consumer_not_allowed',
                    consumers=joined_consumers,
                )
                raise UnknownConsumerError(
                    list(required_grants),
                    ip,
                    joined_consumers,
                    tvm_client_id,
                )
        elif not permission.is_allowed and not permission.is_valid_client_id:
            joined_consumers = ','.join(consumers)
            if parsed_ticket is not None:
                self.graphite_logger.log(
                    status='error',
                    reason='invalid_ticket_source',
                    consumers=joined_consumers,
                    debug_info=parsed_ticket.debug_info(),
                )
                raise InvalidSourceError(ip, joined_consumers, tvm_client_id)
            else:
                self.graphite_logger.log(
                    status='error',
                    reason='missing_ticket',
                    consumers=joined_consumers,
                )
                raise MissingTicketError(ip, joined_consumers, tvm_client_id)
        elif (not permission.is_allowed and
              permission.is_valid_ip and
              permission.missing_required):
            joined_consumers = ','.join(consumers)
            self.graphite_logger.log(
                status='error',
                reason='missing_grants',
                consumers=joined_consumers,
                required_grants=','.join(required_grants),
                missing_grants=','.join(permission.missing_required),
            )
            raise MissingRequiredGrantsError(
                list(permission.missing_required),
                ip,
                joined_consumers,
                tvm_client_id,
            )
        else:
            # TODO: после того, как админы переведут потребителей на макросы,
            # писать логи только в случае ошибок.
            if (not permission.missing_required and
                    required_grants):
                self.graphite_logger.log(
                    status='ok',
                    consumers=','.join(consumers),
                    required_grants=','.join(required_grants),
                )

            if (not permission.missing_required and
                    permission.missing_optional):
                raise MissingOptionalGrantsError(
                    list(permission.missing_optional),
                    ip,
                    ','.join(consumers),
                    tvm_client_id,
                )


def get_grants():
    return LazyLoader.get_instance('Grants')  # pragma: no cover


def check_grant(grant, consumer_ip, consumer, service_ticket=None):
    grants = get_grants()
    grants.check_access(consumer_ip, [consumer], [grant], service_ticket=service_ticket)
