# -*- coding: utf-8 -*-
from collections import namedtuple
import logging
from os import path

from passport.backend.core.conf import settings
from passport.backend.core.dynamic_config import (
    BaseDynamicConfig,
    load_json,
)
from passport.backend.core.grants.exceptions import LoadGrantsError
from passport.backend.core.grants.trypo_compatible_radix import TRYPOCompatibleRadix
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.types.ip.ip import IP
from six import (
    iteritems,
    string_types,
)
import yenv


Permission = namedtuple(
    'Permission',
    'is_allowed is_valid_ip is_valid_client_id missing_required missing_optional',
)

log = logging.getLogger('passport.grants')
optional_log = logging.getLogger('passport.grants.optional')


def get_yenv_type():
    # В rc окружении используем production гранты
    if yenv.type == 'rc':
        type_ = 'production'
    else:
        type_ = yenv.type

    if yenv.name != 'localhost':
        return '%s.%s' % (yenv.name, type_)

    return type_


def check_grant(grant_name, grants, log=log):
    # в случае entity=subscription, precise_grant будет ещё содержать
    # имя сервиса, т.е. входной грант выглядит так 'subscription.create.mail'
    entity, precise_grant = extract_entity_grant(grant_name)

    grants = grants.get(entity)

    # grant-ы могут быть представлены: ['*'], ['precise', ..], {'mail': ['*']},
    # {'mail': ['precise', ..]}
    if isinstance(grants, dict):
        precise_grant, service = extract_entity_grant(precise_grant)
        grants = grants.get(service)

    if not grants:
        log.debug('Empty grants for "%s"', grant_name)
        return False

    if isinstance(grants, string_types):
        grants = [grants]

    if '*' in grants:
        return True

    if not precise_grant:
        log.debug('Grants have not precise grant for grant: "%s"', grant_name)
        return False

    if precise_grant in grants:
        return True

    log.debug('Grant not found: "%s"', grant_name)
    return False


def extract_entity_grant(grant_name):
    if '.' in grant_name:
        return grant_name.split('.', 1)
    else:
        return grant_name, None


def is_test_yandex_login_consumer(consumer):
    """
    Когда-то захотели выдавать разрешения работать с тестовыми логинами без привязки к потребителю,
    однако формат файла с грантами, ориентированный на потребителей, не изменился. Вместо этого был добавлен
    набор скрытых consumer-ов, начинающихся с определенного префикса. Эти данные в обычной логике проверки
    грантов не участвуют.
    FIXME: Это костыль, непонятно почему обычный механизм проверки грантов не подошел.
    """
    return consumer.lower().startswith(settings.TEST_YANDEX_LOGIN_CONSUMER_PREFIX)


@lazy_loadable()
class GrantsConfig(BaseDynamicConfig):
    def __init__(self, grants_paths=None, cache_time=None):
        if grants_paths is None:
            grants_paths = []
            for entry in settings.GRANTS_FILES:
                filename = entry['mask'].format(env_type=get_yenv_type())
                grants_paths.append(path.join(entry['grants_dir'], filename))

        super(GrantsConfig, self).__init__(
            config_filenames=grants_paths,
            cache_time=cache_time or settings.GRANTS_CONFIG_CACHE_TIME,
        )
        self.global_rtree = TRYPOCompatibleRadix()
        self.per_consumer_rtree = {}
        self._consumer_cache = []
        self._tvm_client_id_cache = {}

    def read_config_file(self, filename):
        """Получает гранты из файла с JSON-документом."""
        try:
            config = load_json(filename)
        except ValueError as e:
            raise LoadGrantsError(str(e))
        except IOError as e:
            raise LoadGrantsError(u'{msg} ({filename})'.format(
                msg=e.args[1],
                filename=e.filename,
            ))
        log.debug(
            'Grants for environment %s %s have successfully loaded from file %s',
            yenv.name,
            yenv.type,
            filename,
        )
        return config

    def merge_configs(self, configs):
        uniq_consumers = set()
        merged_grants = dict()
        for grants in configs:
            consumers = set(grants.keys())
            if uniq_consumers & consumers:
                raise LoadGrantsError('Grants files have common consumers')
            uniq_consumers.update(consumers)
            merged_grants.update(grants)
        return merged_grants

    def postprocess(self):
        self._consumer_cache = []
        self._tvm_client_id_cache = {}

        # Используем общее radix-дерево для быстрого поиска потребителя по IP
        self.global_rtree = TRYPOCompatibleRadix()
        # Для каждого потребителя строим отдельное дерево для проверки принадлежности IP сетям потребителя.
        # Отдельное дерево нужно на случай неполного пересечения сетей различных потребителей,
        # например, сеть из одного IP, входящая в более крупную сеть другого потребителя.
        self.per_consumer_rtree = {}
        for consumer, consumer_data in iteritems(self.config):
            tvm_client_id = consumer_data.get('client', {}).get('client_id')
            if tvm_client_id:
                self._tvm_client_id_cache[consumer] = int(tvm_client_id)

            networks = consumer_data.get('networks')

            if networks is None:
                continue

            consumer_rtree = self.per_consumer_rtree.setdefault(consumer, TRYPOCompatibleRadix())
            consumer_is_test_yandex_login = is_test_yandex_login_consumer(consumer)
            for network in networks:
                if not consumer_is_test_yandex_login:
                    rnode = self.global_rtree.add(network)
                    rnode.data.setdefault('consumers', set()).add(consumer)
                consumer_rtree.add(network)

    def is_valid_ip(self, ip, consumer):
        ip = IP(ip).normalized

        if consumer not in self.per_consumer_rtree:
            log.debug('Consumer %s not found', consumer)
            return False

        rnode = self.per_consumer_rtree[consumer].search_best(ip)
        return rnode is not None

    def is_permitted(self, required_grants, optional_grants, ip, consumer, tvm_client_id):
        """
        Поиск грантов для consumer-a. Проверка, что ip принадлежит
        данному consumer-у. Если гранты для consumer найдены,
        тогда пытаемся в них найти есть ли требуемый грант для ручки.

        Возвращает именованный кортеж
            is_permitted
                высказывание "действие разрешено",
            is_valid_ip
                "данный потребитель допускается с данного адреса",
            is_valid_client_id
                "данный потребитель допускается с тикетом от данного tvm-приложения",
            missing_required
                список тех обязательных разрешений, которых нет у consumer'а,
            missing_optional
                список тех дополнительных разрешений, которых нет у consumer'а.
        """
        required_grants = set(required_grants)
        optional_grants = set(optional_grants)

        if consumer is None or not self.is_valid_ip(ip, consumer):
            return Permission(
                is_allowed=False,
                is_valid_ip=False,
                is_valid_client_id=False,
                missing_required=required_grants,
                missing_optional=optional_grants,
            )

        consumer_client_id = self.get_tvm_client_id(consumer)
        if consumer_client_id and tvm_client_id != consumer_client_id:
            return Permission(
                is_allowed=False,
                is_valid_ip=True,
                is_valid_client_id=False,
                missing_required=required_grants,
                missing_optional=optional_grants,
            )

        # Проверки по адресу и TVM-тикету пройдена и требуется хотя бы одно разрешение
        grants = self.config.get(consumer, {}).get('grants')
        if not grants and required_grants:
            log.debug('No grants for consumer "%s"', consumer)
            return Permission(
                is_allowed=False,
                is_valid_ip=True,
                is_valid_client_id=True,
                missing_required=required_grants,
                missing_optional=optional_grants,
            )

        missing_required_grants = list(filter(
            lambda grant: not check_grant(grant, grants, log),
            required_grants,
        ))
        missing_optional_grants = list(filter(
            lambda grant: not check_grant(grant, grants, optional_log),
            optional_grants,
        ))

        return Permission(
            is_allowed=not missing_required_grants,
            is_valid_ip=True,
            is_valid_client_id=True,
            missing_required=set(missing_required_grants),
            missing_optional=set(missing_optional_grants),
        )

    def _get_consumers_by_ip(self, ip):
        ip = IP(ip).normalized

        rnode = self.global_rtree.search_best(ip)

        if rnode is None:
            log.debug('Consumer name not found')
            return set()
        else:
            return rnode.data['consumers']

    def get_all_consumers(self):
        """
        Список всех consumer-ов
        """
        if not self._consumer_cache:
            self._consumer_cache = [
                consumer for consumer in self.config.keys()
                if not is_test_yandex_login_consumer(consumer)
            ]
        return self._consumer_cache

    def get_consumers(self, ip, consumer=None,
                      fallback_when_ip_invalid_for_consumer=True):
        """
        Метод для легаси-апи.

        Если consumer-а нет, то ищем по ip-адресу.

        Если consumer есть, но ip невалидный для этого consumer'а и
        fallback_when_ip_invalid_for_consumer == True, то ищем по ip-адресу.

        Возвращает множество потребителей.
        """
        if consumer is None:
            return self._get_consumers_by_ip(ip)
        elif (
            fallback_when_ip_invalid_for_consumer and
            not self.is_valid_ip(ip, consumer)
        ):
            log.debug('Unknown IP address %s for consumer "%s"', ip, consumer)
            return self._get_consumers_by_ip(ip)

        return set([consumer])

    def get_tvm_client_id(self, consumer):
        return self._tvm_client_id_cache.get(consumer)


def get_grants_config():
    grants_config = LazyLoader.get_instance('GrantsConfig')
    grants_config.load()
    return grants_config


def check_specialtest_yandex_login_grant(ip, suffix):
    """
    Функция определяет по ip и по специальному суффиксу (см. TEST_YANDEX_LOGIN_PREFIXES)
    есть ли у данного ip возможность регистрировать тестовые аккаунты.
    """
    test_yandex_login_consumer = settings.TEST_YANDEX_LOGIN_CONSUMER_PREFIX + suffix
    return get_grants_config().is_valid_ip(ip, test_yandex_login_consumer)


def check_substitute_user_ip_grant(ip):
    """
    Функция определяет по ip источника запроса, есть ли у данного источника возможность передавать
    пользовательский ip в специальном заголовке X-YProxy-Header-Ip
    """
    return get_grants_config().is_valid_ip(ip, settings.SUBSTITUTE_USER_IP_CONSUMER)


def check_substitute_consumer_ip_grant(ip):
    """
    Функция определяет по ip источника запроса, есть ли у данного источника возможность передавать
    ip потребителя в специальном заголовке Ya-Consumer-Client-IP
    """
    return get_grants_config().is_valid_ip(ip, settings.SUBSTITUTE_CONSUMER_IP_CONSUMER)


def check_ip_counters_always_empty(ip):
    """
    Определяем, нужно ли для данного ip выключить проверку ip-счетчиков и всегда их пропускать.
    """
    return get_grants_config().is_valid_ip(ip, settings.TEST_COUNTERS_ALWAYS_EMPTY_CONSUMER)


def check_ip_counters_always_full(ip):
    """
    Определяем, нужно ли для данного ip выключить проверку ip-счетчиков и
    никогда их не пропускать, отвечать ошибкой.
    """
    return get_grants_config().is_valid_ip(ip, settings.TEST_COUNTERS_ALWAYS_FULL_CONSUMER)
