# -*- coding: utf-8 -*-
from collections import defaultdict

from netaddr import IPAddress
from radix import Radix


PROJECT_ID_BIT_LENGTH = 32  # до 8 hex-цифр
PROJECT_ID_BIT_START = 64  # с 64 по 95 биты включительно
BITS_AFTER_PROJECT_ID = (128 - PROJECT_ID_BIT_START - PROJECT_ID_BIT_LENGTH)
PROJECT_ID_MASK = 0xffffffff << BITS_AFTER_PROJECT_ID


def is_trypo_network(network):
    return '@' in network


def extract_project_id(ip):
    # Вырезаем нужные биты и переводим их в hex.
    # Описание формата: https://wiki.yandex-team.ru/noc/newnetwork/hbf/projectid/#chtotakoeprojectid
    project_id = (ip.value & PROJECT_ID_MASK) >> BITS_AFTER_PROJECT_ID
    return hex(project_id).replace('0x', '').rstrip('L')


class TRYPOCompatibleRadix(object):
    def __init__(self):
        self._radix = Radix()
        self._per_project_radixes = defaultdict(Radix)

    def add(self, network):
        if is_trypo_network(network):
            project_id, netmask = network.split('@', 1)
            project_id = project_id.lstrip('0')
            return self._per_project_radixes[project_id].add(netmask)
        else:
            return self._radix.add(network)

    def search_best(self, ip):
        # Нужно поискать в двух местах: в дереве стандартных сетей и в дереве для project_id, записанного в IP.
        address = IPAddress(ip)
        old_style_node = self._radix.search_best(ip)
        new_style_node = None
        if address.version == 6:
            expected_project_id = extract_project_id(address)
            if expected_project_id in self._per_project_radixes:
                new_style_node = self._per_project_radixes[expected_project_id].search_best(ip)

        if (
            not old_style_node or
            (new_style_node and new_style_node.prefixlen + PROJECT_ID_BIT_LENGTH >= old_style_node.prefixlen)
        ):
            return new_style_node
        return old_style_node
