# -*- coding: utf-8 -*-
# представляет события из ручки /events в виде ленты событий
from collections import namedtuple
from datetime import (
    datetime,
    timedelta,
)
import itertools
import logging
import math
import time

from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.conf import settings
from passport.backend.core.historydb.account_history.event_parsers import parse_event
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger('passport.historydb.account_history.events')


DEFAULT_EVENTS_LIMIT = 20  # взято с потолка, сколько будет отдаваться попарсенных событий
HISTORYDB_API_QEURY_LIMIT = 2500  # взято эмпирически, чтобы from_timestamp проростал


HALF_A_YEAR_AGO = settings.TIMESTAMP_DELTA_ONE_DAY * 365 / 2


EventEntry = namedtuple('EventEntry', ['timestamp', 'client_name', 'data', 'geo_id', 'user_ip', 'as_list'])


class AccountHistory(object):
    def __init__(self, uid):
        self.uid = uid
        self.next_page_timestamp = None

    def list(self, to_ts=None, limit=DEFAULT_EVENTS_LIMIT, hours_limit=None):
        historydb_to_ts = int(math.ceil(to_ts)) if to_ts is not None else None
        if hours_limit is not None and hours_limit * 3600 <= HALF_A_YEAR_AGO:
            historydb_from_ts = datetime_to_integer_unixtime(datetime.now() - timedelta(hours=hours_limit))
        else:
            historydb_from_ts = None
        events = self._load_events(from_ts=historydb_from_ts, to_ts=historydb_to_ts, limit=HISTORYDB_API_QEURY_LIMIT)
        groupped_events = self._group_events(events)
        parsed_events = []

        self.next_page_timestamp = None
        for event in groupped_events:
            parsed_event = self._parse_event(event)
            if parsed_event:
                if to_ts is not None and parsed_event.timestamp > to_ts:
                    continue
                if parsed_event.event_type is None:
                    self._mask_fields(event)
                    log.warning('Parsed unknown historydb event: %r', event)
                    continue
                parsed_events.append(parsed_event)

            if len(parsed_events) > limit:
                self.next_page_timestamp = parsed_events[limit].timestamp
                break

        return parsed_events[:limit]

    def _parse_event(self, event):
        return parse_event(event)

    def _mask_fields(self, event):
        for key in event.data:
            if key != 'action' and event.data.get(key):
                event.data[key] = '***'

    def _load_events(self, limit, from_ts=None, to_ts=None):
        historydb = get_historydb_api()
        if from_ts is None:
            half_a_year_ago = datetime.now() - timedelta(seconds=HALF_A_YEAR_AGO)
            from_ts = datetime_to_integer_unixtime(half_a_year_ago)
        if to_ts is None:
            to_ts = int(time.time())

        events = historydb.events(
            self.uid,
            from_ts=from_ts,
            to_ts=to_ts,
            limit=limit,
        )
        return events

    def _group_events(self, events):
        def key_f(event):
            return event['timestamp'], event.get('host_id'), event.get('client_name')
        for key, group in itertools.groupby(sorted(events, key=key_f, reverse=True), key=key_f):
            timestamp = key[0]
            client_name = key[2]
            as_list = None
            geo_id = None
            user_ip = None
            data = {}
            for event in group:
                as_list = as_list or event.get('ip.as_list', '').split(',')
                geo_id = geo_id or event.get('ip.geoid')
                user_ip = user_ip or event.get('user_ip')
                data[event['name']] = event.get('value')
            yield EventEntry(
                timestamp=timestamp,
                client_name=client_name,
                data=data,
                geo_id=int(geo_id) if geo_id is not None else None,
                user_ip=user_ip,
                as_list=as_list,
            )
