# -*- coding: utf-8 -*-
from collections import namedtuple

from netaddr.core import AddrFormatError
from passport.backend.core.env.env import parse_uatraits_value
from passport.backend.core.exceptions import InvalidIpHeaderError
from passport.backend.core.portallib import (
    get_uatraits,
    is_yandex_server_ip,
)
from passport.backend.core.types.ip.ip import IP
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
import six


BrowserEntry = namedtuple('BrowserEntry', ['version', 'name'])
OSEntry = namedtuple('OSEntry', ['version', 'name'])
IPEntry = namedtuple('IPEntry', ['ip', 'AS', 'geoid'])


def filter_loopback(user_ip):
    try:
        if user_ip and IP(user_ip).is_loopback:
            return
        else:
            return user_ip
    except (AddrFormatError, ValueError):
        return


def filter_yandex_server(user_ip):
    try:
        if user_ip and not is_yandex_server_ip(str(user_ip)):
            return user_ip
    except InvalidIpHeaderError:
        return


def filter_ip_address(user_ip):
    return filter_yandex_server(
        filter_loopback(
            user_ip,
        ),
    )


def determine_event_type(parsed_event, raw_event):
    # К сожалению порядок имеет значение для некоторых типов.
    action_types = {a['type'] for a in parsed_event.actions}
    action = raw_event.data.get('action')
    if action and action.startswith('account_register'):
        return
    login_rule = raw_event.data.get('sid.login_rule')
    if login_rule == '8|1' and {'password_change', 'global_logout'} & action_types and 'restore' not in action_types:
        return 'forced_password_change'
    # тайное знание, но так выглядит сброс пользовательских почт, телефонов и т.п.
    elif action == 'restore_entities_flushed':
        return 'restore_entities_flushed'
    elif 'personal_data' in action_types:
        return 'personal_data'
    elif 'restore' in action_types:
        return 'restore'
    elif 'secure_phone_set' in action_types:
        return 'secure_phone_set'
    elif 'secure_phone_unset' in action_types:
        return 'secure_phone_unset'
    elif 'secure_phone_replace' in action_types:
        return 'secure_phone_replace'
    elif 'email_add' in action_types:
        return 'email_add'
    elif {'email_remove', 'email_remove_all'} & action_types:
        return 'email_remove'
    # Смена пароля встречается в событиях про totp как побочный эффект
    elif {'password_change', 'password_remove'} & action_types and 'otp' not in raw_event.data.get('action', ''):
        return 'password'
    elif 'totp_enabled' in action_types:
        return 'totp_enabled'
    elif 'totp_migrated' in action_types:
        return 'totp_migrated'
    elif 'totp_disabled' in action_types:
        return 'totp_disabled'
    elif 'app_passwords_enabled' in action_types:
        return 'app_passwords_enabled'
    elif 'app_passwords_disabled' in action_types:
        return 'app_passwords_disabled'
    elif {'global_logout', 'web_sessions_revoked', 'tokens_revoked', 'app_passwords_revoked'} & action_types:
        return 'global_logout'
    elif {'questions_change', 'questions_remove'} & action_types:
        return 'questions'
    elif 'phone_bind' in action_types:
        return 'phone_bind'
    elif 'phone_unbind' in action_types:
        return 'phone_unbind'


class HistoryDbParsedEvent(object):
    def __init__(self):
        self.event_type = None
        self.timestamp = None
        self.ip = IPEntry(ip=None, AS=None, geoid=None)
        self.browser = BrowserEntry(version=None, name=None)
        self.os = BrowserEntry(version=None, name=None)
        self.actions = []

    @classmethod
    def from_historydb_event(cls, historydb_event):
        event = cls()
        event.timestamp = historydb_event.timestamp

        as_list = []
        for as_name in historydb_event.as_list:
            if as_name.startswith('AS') and len(as_name) > 2:
                as_list.append(int(as_name[2:]))

        as_id = sorted(as_list)[0] if as_list else None
        user_ip = filter_ip_address(historydb_event.user_ip)
        if user_ip:
            event.ip = IPEntry(ip=user_ip, AS=as_id, geoid=historydb_event.geo_id)
        else:
            event.ip = IPEntry(ip=None, AS=None, geoid=None)

        user_agent = historydb_event.data.get('user_agent')
        if user_agent:
            # в обоих питонах detect принимает str,
            # поэтому гарантируем str
            if six.PY2:
                user_agent = smart_bytes(user_agent)
            else:
                user_agent = smart_text(user_agent)
            ua_traits = get_uatraits().detect(user_agent)
            event.browser = BrowserEntry(
                name=parse_uatraits_value(ua_traits.get('BrowserName')),
                version=parse_uatraits_value(ua_traits.get('BrowserVersion')),
            )
            event.os = OSEntry(
                name=parse_uatraits_value(ua_traits.get('OSName')),
                version=parse_uatraits_value(ua_traits.get('OSVersion')),
            )

        return event

    def _asdict(self):
        return {
            'timestamp': self.timestamp,
            'event_type': self.event_type,
            'ip': dict(self.ip._asdict()),
            'os': dict(self.os._asdict()),
            'browser': dict(self.browser._asdict()),
            'actions': self.actions,
        }

    def __repr__(self):
        return '<{} timestamp={} event_type={} actions={}>'.format(
            self.__class__.__name__,
            self.timestamp,
            self.event_type,
            repr(self.actions),
        )
