# -*- coding: utf-8 -*-
import re

from passport.backend.core.types.email.email import (
    mask_email_for_challenge,
    unicode_email,
)


RE_EMAIL = re.compile(r'^email.(\d+)$')


class EmailOldAddEventParser(object):
    def parse(self, raw_event):
        email_line = raw_event.data.get('email.unsafe')
        if not email_line:
            return

        email_address = email_line.split()[0]
        if email_address:
            return {
                'type': 'email_add',
                'email_bind': mask_email_for_challenge(unicode_email(email_address)),
            }


class EmailOldDeleteEventParser(object):
    def parse(self, raw_event):
        email_line = raw_event.data.get('email.delete')
        if not email_line or not email_line.strip():
            return

        email_address = email_line.split(' ')[0]
        if not email_address:
            return
        # удалились все адреса
        if email_address == '*':
            return {
                'type': 'email_remove_all',
            }
        else:
            return {
                'type': 'email_remove',
                'email_unbind': mask_email_for_challenge(unicode_email(email_address)),
            }


class EmailRestoreFlushEventParser(object):
    def parse(self, raw_event):
        if raw_event.data.get('action') != 'restore_entities_flushed':
            return
        # Предполагается, что с новым почтовым валидатором это событие будет избыточно,
        # т.к. сериализатор будет дописывать удаление каждого имейла отдельно в это событие.
        # Поэтому, если есть упоминание имейла в словаре, приходим к выводу, что новый
        # валидатор таки уже в бою и это сообщение выводить не нужно.
        new_email_validator = any('email' in key for key in raw_event.data)
        if not new_email_validator and 'emails' in raw_event.data.get('info.flushed_entities', '').split(','):
            return {
                'type': 'email_remove_all',
            }


class EmailNewEventParser(object):
    def parse(self, raw_event):
        retval = None

        email_ids = set()
        for key in raw_event.data:
            m = RE_EMAIL.match(key)
            if m:
                email_id = m.groups()[0]
                email_ids.add(email_id)
        for email_id in email_ids:
            attr_key = 'email.{}'.format(email_id)
            action = raw_event.data.get(attr_key)
            email_address = raw_event.data.get(attr_key + '.address')
            if action == 'created' and email_address:
                if raw_event.data.get(attr_key + '.created_at'):
                    retval = {
                        'type': 'email_add',
                        'email_bind': mask_email_for_challenge(unicode_email(email_address)),
                    }
            elif action == 'deleted' and email_address:
                retval = {
                    'type': 'email_remove',
                    'email_unbind': mask_email_for_challenge(unicode_email(email_address)),
                }
        return retval
