# -*- coding: utf-8 -*-
import re

from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    mask_phone_number,
    PhoneNumber,
)
from six import iteritems


RE_PHONE_ID_NUMBER = re.compile(r'^phone\.(\d+)\.number$')
RE_PHONE_ID_ACTION = re.compile(r'^phone\.(\d+)\.action$')


def parse_and_mask(phone_number):
    try:
        parsed_phone_number = PhoneNumber.parse(phone_number, allow_impossible=True)
        return mask_phone_number(parsed_phone_number.international)
    except InvalidPhoneNumber:
        return mask_phone_number(phone_number)


def _filter_by_regex(event, regex):
    ids_to_numbers = {}
    for key, value in iteritems(event):
        match = regex.match(key)
        if match:
            phone_id = match.groups()[0]
            ids_to_numbers[phone_id] = value
    return ids_to_numbers


def extract_phone_numbers(event):
    return _filter_by_regex(event, RE_PHONE_ID_NUMBER)


def extract_phone_actions(event):
    return _filter_by_regex(event, RE_PHONE_ID_ACTION)


def get_operation_key_by_phone_id(phone_id, event, operation_key):
    retval = []
    re_pattern = re.compile(r'^phone.{}.operation.\d+.{}$'.format(phone_id, operation_key))
    for key, value in event.items():
        m = re_pattern.match(key)
        if m:
            retval.append(value)
    return retval


def get_other_phone_number(current_phone_id, ids_to_numbers):
    if len(ids_to_numbers) == 1:
        return
    # ключ всегда есть, т.к. сам current_phone_id берётся из того же ids_to_numbers
    ids_to_numbers.pop(current_phone_id)
    return list(ids_to_numbers.values())[0]


class PhoneEventParser(object):
    def parse(self, raw_event):
        retval = {}

        ids_to_numbers = [(id_, parse_and_mask(phone)) for id_, phone in extract_phone_numbers(raw_event.data).items()]

        phones_secure = raw_event.data.get('phones.secure')

        if phones_secure == '0':
            retval['type'] = 'secure_phone_unset'
            if ids_to_numbers:
                retval['phone_unset'] = ids_to_numbers[0][1]
        elif phones_secure:
            masked_phone_number = dict(ids_to_numbers).get(phones_secure)
            if masked_phone_number:
                retval['type'] = 'secure_phone_set'
                retval['phone_set'] = masked_phone_number
                other_phone_number = get_other_phone_number(phones_secure, dict(ids_to_numbers))
                if other_phone_number:
                    retval['type'] = 'secure_phone_replace'
                    retval['phone_unset'] = other_phone_number
        elif len(ids_to_numbers) == 1:
            # обычные, несекьюрные телефонные номера
            ids_to_actions = extract_phone_actions(raw_event.data)
            phone_id, number = ids_to_numbers[0]

            types = get_operation_key_by_phone_id(phone_id, raw_event.data, 'type')
            actions = get_operation_key_by_phone_id(phone_id, raw_event.data, 'action')
            security_identities = get_operation_key_by_phone_id(phone_id, raw_event.data, 'security_identity')
            # в этом парсере избегаем секьюрных телефонов
            if '1' in security_identities:
                return

            if 'bind' in types and 'deleted' in actions and ids_to_actions.get(phone_id) == 'changed':
                retval = {
                    'type': 'phone_bind',
                    'phone_bind': number,
                }
            elif 'mark' in types and 'deleted' in actions and ids_to_actions.get(phone_id) == 'deleted':
                retval = {
                    'type': 'phone_unbind',
                    'phone_unbind': number,
                }
            return retval or None

        # замена телефона с карантином
        if not retval:
            if len(ids_to_numbers) == 2:
                # Что здесь происходит и почему:
                # У кода стоит цель, найти телефон, который будет поставлен как секьюрный после карантина.
                # Известно, что в логе historydb текущий секьюрный номер помечается как security_identity=1.
                # Соответственно, мы находим его из двух представленых номеров, и помечаем его как phone_unset,
                # потому что после карантина этот номер перестанет быть секьюрным.
                # Отсюда же следует, что телефонный номер с security_identity != 1 сейчас не является секьюрным,
                # но после карантина станет секьюрным, поэтому записываем его к phone_set.
                currently_secure = None
                delayed_until = None
                for i, (phone_id, number) in enumerate(ids_to_numbers):
                    security_identities = get_operation_key_by_phone_id(phone_id, raw_event.data, 'security_identity')
                    finished = get_operation_key_by_phone_id(phone_id, raw_event.data, 'finished')
                    if not security_identities or not finished:
                        break
                    if not finished[0].isdigit():
                        break
                    delayed_until = int(finished[0])
                    if '1' in security_identities:
                        # секьюрный телефон может быть только один, поэтому здесь безопасно выйти из цикла
                        currently_secure = i
                        break

                if currently_secure is not None:
                    old_number = ids_to_numbers[currently_secure][1]
                    new_number = ids_to_numbers[(currently_secure + 1) % 2][1]

                    retval.update({
                        'type': 'secure_phone_replace',
                        'phone_set': new_number,
                        'phone_unset': old_number,
                        'delayed_until': delayed_until,
                    })

        reason_uid = raw_event.data.get('reason_uid')
        if retval and reason_uid and reason_uid.isdigit():
            retval['reason_uid'] = int(reason_uid)
            if len(ids_to_numbers) == 1:
                retval['phone_unset'] = ids_to_numbers[0][1]

        return retval or None
