# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import (
    get_origin_info_from_event,
    is_action_account_create_event,
    is_action_delete_event,
    is_action_restore_support_link_created_event,
    is_info_ena_disabled_event,
    is_info_ena_enabled_event,
    is_info_ena_event,
    is_karma_event,
    is_karma_prefix_event,
)
from passport.backend.core.historydb.events import (
    EVENT_ACTION,
    EVENT_INFO_ENA,
    EVENT_INFO_KARMA,
    EVENT_INFO_KARMA_FULL,
    EVENT_INFO_KARMA_PREFIX,
    EVENT_USER_AGENT,
    EVENT_USERINFO_FT,
)
from six import iteritems


log = logging.getLogger('passport.historydb.analyzer.event_handlers.account_state')

# Аккаунт заблокирован саппортом
ACCOUNT_STATUS_DISABLED = 'disabled'

# Аккаунт заблокирован при попытке удаления пользователем
ACCOUNT_STATUS_DISABLED_ON_DELETE = 'disabled_on_delete'

# Аккаунт удален саппортом
ACCOUNT_STATUS_DELETED_BY_SUPPORT = 'deleted_by_support'

# Аккаунт удален пользователем
ACCOUNT_STATUS_DELETED = 'deleted'

# Аккаунт активен, в прошлом был заблокирован
ACCOUNT_STATUS_LIVE_UNBLOCKED = 'live_unblocked'

# Аккаунт активен
ACCOUNT_STATUS_LIVE = 'live'


class RegistrationEnvHandler(EventHandler):
    events = (
        EVENT_ACTION,
        EVENT_USERINFO_FT,
        EVENT_USER_AGENT,
    )

    def __init__(self, *args, **kwargs):
        self.ts_to_reg_info = OrderedDict()
        super(RegistrationEnvHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        if (is_action_account_create_event(event) or event['name'] == EVENT_USERINFO_FT) and event.get('user_ip'):
            ts = event['timestamp']
            reg_info = self.ts_to_reg_info.setdefault(ts, {})
            reg_info.update(get_origin_info_from_event(event))
        elif event['name'] == EVENT_USER_AGENT and event.get('value'):
            reg_info = self.ts_to_reg_info.setdefault(event['timestamp'], {})
            reg_info['user_agent'] = event['value']

    def post_process_events(self):
        reg_infos = []
        for ts, info in iteritems(self.ts_to_reg_info):
            if 'user_ip' in info:
                # Отфильтровываем записи, не содержащие информации о регистрации
                reg_infos.append(info)
        if len(reg_infos) > 1:
            log.warning('Detected multiple registration events: %s', reg_infos)
        if reg_infos:
            return dict(registration_env=reg_infos[0])


class AccountCreateDeleteHandler(EventHandler):
    """
    Создание, удаление аккаунта
    """
    events = (
        EVENT_ACTION,
        EVENT_USERINFO_FT,
    )

    def __init__(self, *args, **kwargs):
        self.create_event = self.delete_event = None
        super(AccountCreateDeleteHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        if is_action_account_create_event(event) or event['name'] == EVENT_USERINFO_FT:
            if self.create_event:
                log.warning('Multiple account create events: %s %s', self.create_event, event)
            self.create_event = event
        elif is_action_delete_event(event):
            if self.delete_event:
                log.warning('Multiple account delete events: %s %s', self.delete_event, event)
            self.delete_event = event

    def post_process_events(self):
        return dict(account_create_event=self.create_event, account_delete_event=self.delete_event)


def _fill_result_with_event_metadata(result, event):
    result['admin'] = event.get('admin')
    result['comment'] = event.get('comment')
    result['user_ip'] = event.get('user_ip')


class AccountEnabledStatusHandler(EventHandler):
    """
    Статус аккаунта и блокировки
    """
    events = (
        EVENT_ACTION,
        EVENT_INFO_KARMA,
        EVENT_INFO_KARMA_FULL,
        EVENT_INFO_KARMA_PREFIX,
        EVENT_INFO_ENA,
    )

    def __init__(self, *args, **kwargs):
        self.ts_to_events = OrderedDict()
        self.last_disabled_event = None
        self.karma_events = []
        super(AccountEnabledStatusHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        if (is_action_delete_event(event) or
                is_info_ena_event(event) or
                is_action_restore_support_link_created_event(event)):
            self.ts_to_events.setdefault(event['timestamp'], {})[event['name']] = event
        elif is_karma_event(event) or is_karma_prefix_event(event):
            self.karma_events.append(event)

        if is_info_ena_disabled_event(event):
            self.last_disabled_event = event

    def _get_karma_info(self):
        result = []
        for event in self.karma_events:
            value = int(event.get('value') or 0)
            info = dict(
                timestamp=event['timestamp'],
                admin=event.get('admin'),
                comment=event.get('comment'),
            )
            if is_karma_event(event):
                info['karma'] = value
            else:
                info['karma_prefix'] = value
            result.append(info)
        return result

    def post_process_events(self):
        result = {'karma_events': self._get_karma_info()}

        for ts, events in reversed(self.ts_to_events.items()):
            action = events.get(EVENT_ACTION)
            info_ena_event = events.get(EVENT_INFO_ENA)
            deleted_event = is_action_delete_event(action) and action
            is_disabled = is_info_ena_disabled_event(info_ena_event)
            is_enabled = is_info_ena_enabled_event(info_ena_event)

            if deleted_event and is_disabled:
                # Заблокирован при удалении.
                result['status'] = ACCOUNT_STATUS_DISABLED_ON_DELETE
                result['user_ip'] = deleted_event.get('user_ip') or info_ena_event.get('user_ip')
            elif deleted_event:
                # Удален: самостоятельно или саппортом
                if deleted_event.get('admin'):
                    # Удален саппортом
                    result['status'] = ACCOUNT_STATUS_DELETED_BY_SUPPORT
                    _fill_result_with_event_metadata(result, deleted_event)
                else:
                    result['status'] = ACCOUNT_STATUS_DELETED
                    result['user_ip'] = deleted_event.get('user_ip')
            elif is_disabled:
                # Заблокирован саппортом
                result['status'] = ACCOUNT_STATUS_DISABLED
                _fill_result_with_event_metadata(result, info_ena_event)
                if is_action_restore_support_link_created_event(action):
                    # Информация о саппорте при генерации ссылки на восстановлении пишется в событии action
                    _fill_result_with_event_metadata(result, action)
            elif is_enabled and self.last_disabled_event:
                # Аккаунт разблокирован, когда-то был заблокирован
                result['status'] = ACCOUNT_STATUS_LIVE_UNBLOCKED
                _fill_result_with_event_metadata(result, info_ena_event)

            if deleted_event or (info_ena_event and self.last_disabled_event):
                # Выходим, т.к. нашли интересующее нас условие:
                # - аккаунт удален
                # - аккаунт когда-либо был заброкирован и на текущий момент заблокирован или активен
                result['timestamp'] = ts
                break
        else:
            # Аккаунт активен, блокировок не было
            result['status'] = ACCOUNT_STATUS_LIVE

        if self.ts_to_events:
            last_event_ts = list(self.ts_to_events.keys())[-1]
            if (result['status'] not in (ACCOUNT_STATUS_LIVE, ACCOUNT_STATUS_LIVE_UNBLOCKED) and
                    last_event_ts != result['timestamp']):
                log.warning('Unexpected events after account status "%s"', result['status'])

        return dict(account_enabled_status=result)
