# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging
import re

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import (
    get_origin_info_from_event,
    serialize_info_from_ordered_dict,
)
from passport.backend.core.historydb.events import (
    ACTIONS_DELETE_ENTITIES_BY_SUPPORT_LINK_PERL,
    EVENT_ACTION,
    EVENT_EMAIL_CONFIRM,
    EVENT_EMAIL_DELETE,
    EVENT_EMAIL_PREFIX,
    EVENT_EMAIL_RPOP,
    EVENT_INFO_FLUSHED_ENTITIES,
    EVENT_USER_AGENT,
)
from six import (
    iteritems,
    itervalues,
)


log = logging.getLogger('passport.historydb.analyzer.event_handlers.email')


class EmailEventGroup(object):
    """
    Аккумулирует знания о действиях с email-ами в истории в один момент времени
    """
    EMAIL_EVENT_RE = re.compile(r'^email\.(?P<email_id>\d+)(\.(?P<attribute>\w+))?$')

    def __init__(self):
        self.other_events = {}
        self.emails = {}
        self.origin_info = {}
        self.is_global_delete = False

    def add_event(self, event):
        # Собираем origin_info
        if event['name'] == EVENT_USER_AGENT:
            self.origin_info['user_agent'] = event.get('value')
            return
        self.origin_info.update(get_origin_info_from_event(event))

        # События в новой схеме email-валидатора требуют преобразования
        event_name = event['name']
        match = self.EMAIL_EVENT_RE.match(event_name)
        if match:
            email_info = self.emails.setdefault(match.group('email_id'), {})
            attribute = match.group('attribute')
            if attribute:
                email_attrs = email_info.setdefault('attributes', {})
                email_attrs[attribute] = event['value']
            else:
                # Событие вида email.id: deleted
                email_info['action'] = event['value']
        elif (
            event_name == EVENT_INFO_FLUSHED_ENTITIES and 'emails' in event['value'].split(',') or
            event_name == EVENT_ACTION and event['value'] in ACTIONS_DELETE_ENTITIES_BY_SUPPORT_LINK_PERL
        ):
            # Удаление всех адресов - при этом других событий может не быть
            self.is_global_delete = True
        else:
            self.other_events[event_name] = event.get('value')

    def get_confirmed_status_changes(self):
        for email_id, email_info in iteritems(self.emails):
            # По соглашению, для любого действия с адресом в новом валидаторе пишется событие email.id.address
            email_attrs = email_info.get('attributes', {})
            if 'address' not in email_attrs:
                # Нет адреса - такое может быть в тестинге, адрес не сразу начали записывать при любых изменениях
                log.warning('No email address in events for ID %s', email_id)
                continue
            address = email_attrs['address']

            if 'confirmed_at' in email_attrs:
                confirmed_at = email_attrs['confirmed_at']
                yield address, bool(confirmed_at)
            elif email_info.get('action') == 'deleted':
                # При удалении всего адреса не пишутся удаляемые атрибуты
                yield address, False

        if not self.emails:
            if self.is_global_delete:
                # "Глобальное" удаление рассматриваем только если работаем со старой схемой валидатора,
                # т.е. нет новых событий вида email.id.attribute: value
                yield None, False
            for event_name, value in iteritems(self.other_events):
                # События старого валидатора
                if event_name == EVENT_EMAIL_CONFIRM:
                    address = value.split()[0]
                    yield address, True
                elif event_name == EVENT_EMAIL_RPOP:
                    address, action = value.split()
                    if action == 'added':
                        yield address, True
                elif event_name == EVENT_EMAIL_DELETE:
                    yield value, False


class ConfirmedEmailEventHandler(EventHandler):
    events = (
        EVENT_ACTION,
        EVENT_USER_AGENT,
        EVENT_EMAIL_PREFIX,
        # События старого валидатора
        EVENT_EMAIL_CONFIRM,
        EVENT_EMAIL_RPOP,
        EVENT_EMAIL_DELETE,
        # Удаление по ссылке (новое восстановление)
        EVENT_INFO_FLUSHED_ENTITIES,
    )

    def __init__(self, *args, **kwargs):
        self.email_event_groups = OrderedDict()
        self.emails = OrderedDict()
        super(ConfirmedEmailEventHandler, self).__init__(*args, **kwargs)

    def _confirm_email(self, email, origin_info):
        interval = {'start': origin_info, 'end': None}
        if email not in self.emails:
            self.emails[email] = {'intervals': [interval]}
        else:
            email_info = self.emails[email]
            last_interval = email_info['intervals'][-1]
            if last_interval['end'] is not None:
                # это не повторное подтверждение того же адреса
                email_info['intervals'].append(interval)

    def _unconfirm_email(self, email, origin_info):
        if email not in self.emails:
            # удаление неподтвержденного адреса,
            # либо через валидатор удаляется адрес-сборщик, настроенный по IMAP - для таких сборщиков нет
            # информации о добавлении DARIA-44500,
            # либо мы посмотрели недостаточно глубоко и не увидели привязки адреса
            return
        email_info = self.emails[email]
        last_interval = email_info['intervals'][-1]
        if last_interval['end'] is None:
            last_interval['end'] = origin_info
        else:
            # адрес когда-то был потвержден, после чего удален, после чего был начат процесс подтверждения,
            # либо ситуация, аналогичная описанной в комментарии выше
            pass

    def handle_event(self, event):
        event_group = self.email_event_groups.setdefault(event['timestamp'], EmailEventGroup())
        event_group.add_event(event)

    def post_process_events(self):
        for event_group in itervalues(self.email_event_groups):
            for target, is_confirmed in event_group.get_confirmed_status_changes():
                target_emails = [target]
                if target is None:
                    # Удаление всех адресов единым событием
                    target_emails = self.emails.keys()
                for email in target_emails:
                    if is_confirmed:
                        self._confirm_email(email, event_group.origin_info)
                    else:
                        self._unconfirm_email(email, event_group.origin_info)
        return dict(confirmed_emails=serialize_info_from_ordered_dict(self.emails))
