# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import get_origin_info_from_event
from passport.backend.core.historydb.events import (
    EVENT_ACTION,
    EVENT_APP_KEY_INFO,
    EVENT_INFO_BIRTHDAY,
    EVENT_INFO_FIRSTNAME,
    EVENT_INFO_HINTA,
    EVENT_INFO_HINTQ,
    EVENT_INFO_LASTNAME,
    EVENT_INFO_PASSWORD,
    EVENT_USER_AGENT,
    EVENT_USERINFO_FT,
)
from passport.backend.core.types.birthday import Birthday
from six import iteritems


log = logging.getLogger('passport.historydb.analyzer.event_handlers')


EVENT_NAME_MAPPING = {
    'hintq': 'questions',
    'hinta': 'answers',
    'password_hash': 'password_hashes',
    'app_key_info': 'app_key_info',
}


class SimpleEventHandler(EventHandler):
    def __init__(self, *args, **kwargs):
        self.collected_values = {}

        self.event_handlers = {
            EVENT_INFO_FIRSTNAME: self._simple_value_field_handler('firstname'),
            EVENT_INFO_LASTNAME: self._simple_value_field_handler('lastname'),
            EVENT_USERINFO_FT: self._userinfo_ft_handler,
            EVENT_INFO_HINTA: self._simple_value_field_handler('hinta'),
            EVENT_INFO_HINTQ: self._simple_value_field_handler('hintq'),
            EVENT_INFO_PASSWORD: self._simple_value_field_handler('password_hash'),
            EVENT_APP_KEY_INFO: self._simple_value_field_handler('app_key_info'),
        }

        super(SimpleEventHandler, self).__init__(*args, **kwargs)

    def _update_field_value(self, field, value):
        field_key = EVENT_NAME_MAPPING.get(field, field + 's')
        self.collected_values.setdefault(field_key, []).append(value)

    def _userinfo_ft_handler(self, event):
        for field in ('firstname', 'lastname', 'hintq', 'hinta'):
            if event.get(field):
                self._update_field_value(field, event[field])

    def _simple_value_field_handler(self, field):
        def _handler(event):
            if event.get('value'):
                self._update_field_value(field, event['value'])
        return _handler

    def handle_event(self, event):
        handler = self.event_handlers.get(event['name'])
        if handler:
            handler(event)

    def post_process_events(self):
        return self.collected_values


class NamesEventHandler(EventHandler):
    """
    Обработчик группирует события установки имени и фамилии в пары (возможны повторы), сохраняет информацию о
    времени и окружении обновления.
    """
    events = (
        EVENT_USERINFO_FT,
        EVENT_INFO_FIRSTNAME,
        EVENT_INFO_LASTNAME,
        EVENT_USER_AGENT,
        EVENT_ACTION,
    )

    def __init__(self, *args, **kwargs):
        self.ts_to_names_info = OrderedDict()
        self.ts_to_origin_info = {}
        super(NamesEventHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        timestamp = event['timestamp']
        origin_info = self.ts_to_origin_info.setdefault(timestamp, {})
        if event['name'] == EVENT_USER_AGENT:
            origin_info['user_agent'] = event.get('value')
        elif event['name'] == EVENT_ACTION:
            origin_info.update(get_origin_info_from_event(event))
        elif event['name'] == EVENT_USERINFO_FT and (event.get('firstname') or event.get('lastname')):
            origin_info.update(get_origin_info_from_event(event))
            names_info = dict(
                firstname=event.get('firstname'),
                lastname=event.get('lastname'),
                interval={'start': origin_info, 'end': None},
            )
            self.ts_to_names_info[timestamp] = names_info
        elif event['name'] in (EVENT_INFO_FIRSTNAME, EVENT_INFO_LASTNAME) and event.get('value'):
            origin_info.update(get_origin_info_from_event(event))
            names_info = self.ts_to_names_info.setdefault(
                timestamp,
                {'firstname': None, 'lastname': None},
            )
            field = 'firstname' if event['name'] == EVENT_INFO_FIRSTNAME else 'lastname'
            names_info[field] = event['value']
            names_info['interval'] = {'start': origin_info, 'end': None}

    def post_process_events(self):
        current_names = {'firstname': None, 'lastname': None}
        prev_names_info = None
        for ts, names_info in iteritems(self.ts_to_names_info):
            for field in ('firstname', 'lastname'):
                if names_info[field] is None and current_names[field]:
                    # Заполняем "пробелы" - случаи, когда изменилось только имя или только фамилия
                    names_info[field] = current_names[field]
                current_names[field] = names_info[field]
            if prev_names_info is not None:
                prev_names_info['interval']['end'] = names_info['interval']['start']
            prev_names_info = names_info
        return dict(names=list(self.ts_to_names_info.values()))


class BirthdayEventHandler(EventHandler):
    """
    Обработчик группирует события установки и удаления даты рождения, сохраняет информацию о
    времени и окружении обновления.
    """
    events = (
        EVENT_INFO_BIRTHDAY,
        EVENT_USER_AGENT,
        EVENT_ACTION,
    )

    def __init__(self, *args, **kwargs):
        self.birthdays = []
        self.ts_to_origin_info = {}
        super(BirthdayEventHandler, self).__init__(*args, **kwargs)

    def _set_birthday(self, birthday, origin_info):
        if self.birthdays:
            last_birthday_info = self.birthdays[-1]
            if last_birthday_info['interval']['end'] is None:
                # нужно завершить интервал актуальности предыдущего ДР
                # интервал мог быть завершен, если пользователь удалил предыдущее ДР вручную
                last_birthday_info['interval']['end'] = origin_info
        self.birthdays.append(dict(
            value=birthday,
            interval=dict(start=origin_info, end=None),
        ))

    def _remove_birthday(self, origin_info):
        if self.birthdays:
            last_birthday_info = self.birthdays[-1]
            if last_birthday_info['interval']['end'] is None:
                last_birthday_info['interval']['end'] = origin_info
                return
        log.warning(u'Unexpected birthday remove event')

    def handle_event(self, event):
        timestamp = event['timestamp']
        origin_info = self.ts_to_origin_info.setdefault(timestamp, {})
        if event['name'] == EVENT_USER_AGENT:
            origin_info['user_agent'] = event.get('value')
        elif event['name'] == EVENT_ACTION:
            origin_info.update(get_origin_info_from_event(event))
        elif event['name'] == EVENT_INFO_BIRTHDAY:
            origin_info.update(get_origin_info_from_event(event))
            if event.get('value'):
                # Устанавливаем значение ДР
                try:
                    birthday = Birthday.parse(event['value'])
                except ValueError as ex:
                    log.warning('HistoryDB birthday validation error: %s', ex)
                else:
                    self._set_birthday(birthday, origin_info)
            else:
                # Ручное удаление ДР (в событии нет значения)
                self._remove_birthday(origin_info)

    def post_process_events(self):
        return dict(birthdays=self.birthdays)
