# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import (
    get_origin_info_from_event,
    is_action_change_password_event,
    is_action_password_event,
    is_action_restore_passed_event,
    is_changing_required_set_event,
    is_changing_required_unset_event,
)
from passport.backend.core.historydb.events import (
    ACTION_RESTORE_PASSED_BY_METHOD_REGEXP,
    EVENT_ACTION,
    EVENT_INFO_PASSWORD,
    EVENT_RESTORE_METHOD_PASSED,
    EVENT_SID_LOGIN_RULE,
    EVENT_USER_AGENT,
    PASSWORD_CHANGE_TYPE_FORCED,
    PASSWORD_CHANGE_TYPE_RESTORE,
    PASSWORD_CHANGE_TYPE_VOLUNTARY,
)
from six import iteritems


log = logging.getLogger('passport.historydb.analyzer.event_handlers.password')


class PasswordChangeHandler(EventHandler):
    """
    Смены пароля, принуждения к сменам
    """
    events = (
        EVENT_ACTION,
        EVENT_SID_LOGIN_RULE,
        EVENT_INFO_PASSWORD,
        EVENT_USER_AGENT,
        EVENT_RESTORE_METHOD_PASSED,
    )

    def __init__(self, *args, **kwargs):
        self.ts_to_events = OrderedDict()
        super(PasswordChangeHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        if (is_changing_required_set_event(event) or
                is_changing_required_unset_event(event) or
                is_action_password_event(event) or
                is_action_change_password_event(event) or
                is_action_restore_passed_event(event) or
                event['name'] in (EVENT_USER_AGENT, EVENT_RESTORE_METHOD_PASSED, EVENT_INFO_PASSWORD)):
            ts_events = self.ts_to_events.setdefault(event['timestamp'], {})
            ts_events[event['name']] = event

    def post_process_events(self):
        password_change_requests = []
        password_changes = []
        for ts, events in iteritems(self.ts_to_events):
            if EVENT_ACTION in events:
                # Сюда приходят только два события - password и change_password
                action_event = events[EVENT_ACTION]
                sid_event = None
                if EVENT_SID_LOGIN_RULE in events:
                    sid_event = events[EVENT_SID_LOGIN_RULE]
                origin_info = get_origin_info_from_event(action_event)
                if EVENT_USER_AGENT in events:
                    origin_info['user_agent'] = events[EVENT_USER_AGENT].get('value')
                if (is_action_password_event(action_event) and sid_event and
                        (is_changing_required_set_event(sid_event) or is_changing_required_unset_event(sid_event))):
                    password_change_requests.append(dict(
                        origin_info=origin_info,
                        admin=action_event.get('admin'),
                        comment=action_event.get('comment'),
                        change_required=is_changing_required_set_event(sid_event),
                    ))
                elif (is_action_change_password_event(action_event) and sid_event and
                        is_changing_required_unset_event(sid_event)):
                    password_changes.append(dict(
                        origin_info=origin_info,
                        change_type=PASSWORD_CHANGE_TYPE_FORCED,
                    ))
                elif is_action_change_password_event(action_event):
                    password_changes.append(dict(
                        origin_info=origin_info,
                        change_type=PASSWORD_CHANGE_TYPE_VOLUNTARY,
                    ))
                elif is_action_restore_passed_event(action_event):
                    # Новое восстановление
                    method = ACTION_RESTORE_PASSED_BY_METHOD_REGEXP.match(action_event['value']).group('method')
                    password_changes.append(dict(
                        origin_info=origin_info,
                        change_type=PASSWORD_CHANGE_TYPE_RESTORE,
                        restore_method=method,
                    ))
            elif EVENT_RESTORE_METHOD_PASSED in events:
                # Старое восстановление (перл)
                method_passed_event = events[EVENT_RESTORE_METHOD_PASSED]
                origin_info = get_origin_info_from_event(method_passed_event)
                if EVENT_USER_AGENT in events:
                    origin_info['user_agent'] = events[EVENT_USER_AGENT].get('value')
                password_changes.append(dict(
                    origin_info=origin_info,
                    change_type=PASSWORD_CHANGE_TYPE_RESTORE,
                    restore_method=method_passed_event['value'],
                ))
            elif EVENT_INFO_PASSWORD in events:
                # случай регистрации аккаунта
                pass

        return dict(
            password_change_requests=password_change_requests,
            password_changes=password_changes,
        )
