# -*- coding: utf-8 -*-
from collections import OrderedDict
from datetime import datetime
import logging
import re

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import (
    get_origin_info_from_event,
    serialize_info_from_ordered_dict,
)
from passport.backend.core.historydb.events import (
    EVENT_ACTION,
    EVENT_OLD_YASMS_PHONE_ACTION,
    EVENT_OLD_YASMS_PHONE_NUMBER,
    EVENT_OLD_YASMS_PHONE_STATUS,
    EVENT_PHONE_PREFIX,
    EVENT_USER_AGENT,
)
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.utils.time import datetime_to_unixtime
from six import itervalues


log = logging.getLogger('passport.historydb.analyzer.event_handlers.phone')


def _is_old_yasms_action_save_event(event):
    return event['name'] == EVENT_OLD_YASMS_PHONE_ACTION and event.get('value') == 'save'


def _is_old_yasms_action_delete_event(event):
    return event['name'] == EVENT_OLD_YASMS_PHONE_ACTION and event.get('value') == 'delete'


def _is_old_yasms_status_valid_event(event):
    return event['name'] == EVENT_OLD_YASMS_PHONE_STATUS and event.get('value') == 'valid'


class PhoneEventGroup(object):
    """
    Аккумулирует знания о действиях с телефонами в истории в один момент времени
    """
    PHONE_EVENT_RE = re.compile(r'^phone\.(?P<phone_id>\d+)\.(?P<attribute>\w+)$')

    def __init__(self):
        self.events = {}
        self.phones = {}
        self.origin_info = {}

    def add_event(self, event):
        # Собираем origin_info
        if EVENT_USER_AGENT == event['name']:
            self.origin_info['user_agent'] = event.get('value')
        else:
            self.origin_info.update(get_origin_info_from_event(event))
        # Телефонные события в новой схеме YaSMS требуют преобразования
        event_name = event['name']
        match = self.PHONE_EVENT_RE.match(event_name)
        if not match:
            self.events[event_name] = event
        else:
            phone_attrs = self.phones.setdefault(match.group('phone_id'), {})
            attribute = match.group('attribute')
            value = event['value']
            if attribute == 'number':
                value = _get_phone_from_event(event)
                if not value:
                    return
            phone_attrs[attribute] = value

    def bound_phones(self):
        """
        Получить привязанные телефоны
        """
        for phone_attrs in itervalues(self.phones):
            if 'bound' in phone_attrs and 'number' in phone_attrs:
                if phone_attrs['bound'] == '0':
                    continue
                yield phone_attrs['number']

        if EVENT_OLD_YASMS_PHONE_NUMBER in self.events:
            # Обрабатываем события старого YaSMS
            phone, action_or_status_event = self._gather_old_yasms_data()
            if not phone or not action_or_status_event:
                return
            if (_is_old_yasms_action_save_event(action_or_status_event) or
                    _is_old_yasms_status_valid_event(action_or_status_event)):
                yield phone

    def unbound_phones(self):
        """
        Получить отвязанные телефоны и информацию об окружении
        """
        for phone_attrs in itervalues(self.phones):
            if 'number' not in phone_attrs:
                continue
            if ('bound' in phone_attrs and phone_attrs['bound'] == '0' or
                    'action' in phone_attrs and phone_attrs['action'] == 'deleted'):
                yield phone_attrs['number']

        if EVENT_OLD_YASMS_PHONE_NUMBER in self.events:
            # Обрабатываем события старого YaSMS
            phone, action_or_status_event = self._gather_old_yasms_data()
            if not phone or not action_or_status_event:
                return
            if _is_old_yasms_action_delete_event(action_or_status_event):
                yield phone

    def _gather_old_yasms_data(self):
        phone_event = self.events[EVENT_OLD_YASMS_PHONE_NUMBER]
        phone = _get_phone_from_event(phone_event)
        action_or_status_event = self.events.get(
            EVENT_OLD_YASMS_PHONE_ACTION,
            self.events.get(EVENT_OLD_YASMS_PHONE_STATUS, None),
        )
        if not action_or_status_event:
            log.debug('No action or status event for old YaSMS number event')
        return phone, action_or_status_event


class PhoneEventHandler(EventHandler):
    events = (
        EVENT_ACTION,  # для получения yandexuid
        EVENT_USER_AGENT,
        EVENT_PHONE_PREFIX,
        # События старого YaSMS
        EVENT_OLD_YASMS_PHONE_ACTION,
        EVENT_OLD_YASMS_PHONE_NUMBER,
        EVENT_OLD_YASMS_PHONE_STATUS,
    )

    def __init__(self, *args, **kwargs):
        self.phone_event_groups = OrderedDict()
        self.confirmed_phones = OrderedDict()
        super(PhoneEventHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        """
        Требуется найти в истории все провалидированные телефоны, а также операции удаления таких телефонов.
        Для этого группируем события по timestamp'у.

        Для старого YaSMS ищем такие события:
        1) yasms.phone.action: save, yasms.phone.number: <number> - подтверждение через ручку
        2) yasms.phone.number: <number>, yasms.phone.status: valid - подтверждение через смс
        3) yasms.phone.action: delete, yasms.phone.number: <number> - удаление через смс
        Из-за бага YaSms, часть событий вида 2) не соответствуют реально подтвержденному
        номеру, их не отличить от корректных событий, поэтому считаем такие номера
        подтвержденными.
        Также в ручке /deletephone YaSMS до 13.02.2015 не писалась запись об удалении номера.

        Для нового YaSMS ищем:
        1) phone.<id>.bound: <timestamp>, phone.<id>.number: <number> - привязка телефона (по определению подтвержденного)
        2) phone.<id>.action: deleted, phone.<id>.number: <number> - удаление
        3) phone.<id>.bound: 0, phone.<id>.number: <number> - удаление по лимиту числа привязок
        """
        event_group = self.phone_event_groups.setdefault(event['timestamp'], PhoneEventGroup())
        event_group.add_event(event)

    def post_process_events(self):
        for event_group in itervalues(self.phone_event_groups):
            for phone in event_group.bound_phones():
                self._bind_phone(phone, event_group.origin_info)
            for phone in event_group.unbound_phones():
                self._unbind_phone(phone, event_group.origin_info)

        return dict(confirmed_phones=serialize_info_from_ordered_dict(self.confirmed_phones))

    def _bind_phone(self, phone, origin_info):
        interval = {'start': origin_info, 'end': None}
        if phone not in self.confirmed_phones:
            self.confirmed_phones[phone] = {'intervals': [interval]}
        else:
            phone_info = self.confirmed_phones[phone]
            last_interval = phone_info['intervals'][-1]
            if last_interval['end'] is not None:
                phone_info['intervals'].append(interval)
            else:
                # В старом YaSMS повторное событие о подтверждении номера - возможно, мы не нашли запись об удалении PASSP-12037
                log.debug(u'Duplicate phone confirmation event')

    def _unbind_phone(self, phone, origin_info):
        if phone not in self.confirmed_phones:
            # В старом YaSMS - спецэффект от того, что историю телефонов не сразу начали писать PASSP-12037
            # В новом YaSMS нормальная ситуация (удаление непривязанного телефона)
            log.debug(u'Trying to delete unknown phone')
            return
        phone_info = self.confirmed_phones[phone]
        last_interval = phone_info['intervals'][-1]
        if last_interval['end'] is None:
            last_interval['end'] = origin_info
        else:
            # В старом YaSMS повторное событие об удалении номера - возможно, мы не нашли запись о подтверждении PASSP-12037
            log.debug(u'Duplicate phone deletion event')


def flatten_phones_mapping(phones):
    """
    Построить список телефонов, отсортированный по времени подтверждения
    """
    return sorted(
        [
            dict(value=phone_info['value'], interval=interval) for phone_info in phones
            for interval in phone_info['intervals']
        ],
        key=lambda item: item['interval']['start']['timestamp'],
    )


# Дата окончательного перехода на новый YaSMS - после этой даты невалидные телефоны в истории
# логгируем как ворнинги
OLD_YASMS_DEATH_TIMESTAMP = datetime_to_unixtime(datetime(2015, 12, 18))


def _get_phone_from_event(event):
    phone = parse_phone_number(event.get('value', ''))
    if not phone:
        timestamp = event['timestamp']
        logger = log.warning
        if timestamp <= OLD_YASMS_DEATH_TIMESTAMP:
            logger = log.debug
        logger(
            u'Invalid phone in event: %s.',
            event,
        )
        return
    return phone.digital
