# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.analyzer.event_handlers.helpers import get_origin_info_from_event
from passport.backend.core.historydb.events import (
    EVENT_ACTION,
    EVENT_INFO_HINTA,
    EVENT_INFO_HINTQ,
    EVENT_USER_AGENT,
    EVENT_USERINFO_FT,
)
from passport.backend.core.types.question import Question
import six
from six import iteritems


log = logging.getLogger('passport.historydb.analyzer.event_handlers.question_answer')


MAX_HINTQ_HINTA_TS_OFFSET = 1e-3


def _get_hintq_if_specified(hintq, language=None):
    """
    Проверка того, что КВ означает задание реального КВ.
    В событиях userinfo_ft встречаются КВ для случая "не выбран" - считаем их пустыми.
    Бывает еще случай, когда пользователь выбрал собственный КВ, но текст вопроса это пустая строка.
    На процесс восстановления такой КВ не должен повлиять, т.к. анкета не пропускает пустые строки в своих полях.
    """
    if hintq:
        question = Question.parse(hintq, language=language)
        if not question.is_empty:
            return question


class QuestionAnswerMappingHandler(EventHandler):
    events = (
        EVENT_USERINFO_FT,
        EVENT_INFO_HINTA,
        EVENT_INFO_HINTQ,
        EVENT_USER_AGENT,
        EVENT_ACTION,
    )

    def __init__(self, *args, **kwargs):
        self.ts_to_question_answer = OrderedDict()
        self.ts_to_origin_info = {}
        self.language = kwargs.pop('language')
        super(QuestionAnswerMappingHandler, self).__init__(*args, **kwargs)

    def _update_ts_to_question_answer_mapping(self, origin_info, hintq=None, hinta=None):
        """
        Обновить отображение timestamp'а в КВ/КО для заданного события, отфильтровать некорректные случаи задания КВ.
        Такое отображение позволяет предварительно сгруппировать парные события КВ/КО в виде, удобном для дальнейшей
        обработки.
        """
        hintq_specified = _get_hintq_if_specified(hintq, self.language)  # игнорируем КВ вида "0:не указано"
        if hintq_specified or hinta:
            ts_question_answer = self.ts_to_question_answer.setdefault(
                origin_info['timestamp'],
                dict(question=None, answer=None),
            )
            if hintq_specified:
                ts_question_answer['question'] = dict(value=six.text_type(hintq_specified), origin_info=origin_info)
            if hinta:
                if ts_question_answer['answer']:
                    log.warning('Multiple answers set at the same moment')
                ts_question_answer['answer'] = dict(value=hinta, origin_info=origin_info)

    def handle_event(self, event):
        """
        При первом проходе событий пытаемся собрать их в пары КВ/КО, сгруппированные по timestamp'у.
        """
        origin_info = self.ts_to_origin_info.setdefault(event['timestamp'], {})
        if event['name'] == EVENT_USER_AGENT:
            origin_info['user_agent'] = event.get('value')
        elif event['name'] == EVENT_ACTION:
            origin_info.update(get_origin_info_from_event(event))
        elif event['name'] == EVENT_USERINFO_FT:
            origin_info.update(get_origin_info_from_event(event))
            self._update_ts_to_question_answer_mapping(
                origin_info,
                hintq=event.get('hintq'),
                hinta=event.get('hinta'),
            )
        elif event['name'] in (EVENT_INFO_HINTQ, EVENT_INFO_HINTA):
            if event.get('value'):
                origin_info.update(get_origin_info_from_event(event))
                field = 'hintq' if event['name'] == EVENT_INFO_HINTQ else 'hinta'
                self._update_ts_to_question_answer_mapping(
                    origin_info,
                    **{field: event['value']}
                )

    def _update_question_answer_intervals(
        self,
        question_to_answers,
        question,
        answer,
        current_answer_info,
        use_origin_info_from_answer=True,
    ):
        """
        Обновить отображение КВ в КО для новых значений КВ question и КО answer.
        Обновить интервалы актуальности для текущего и нового КО.
        Рассматриваются случаи задания новых КВ/КО, нового КО на тот же КВ, нового КВ без изменения КО, задания
        тех же значений КВ/КО.
        @param question_to_answers: формируемое отображение КВ в КО.
        @param question: значение КВ в формате
            dict(value='new question', origin_info=dict(timestamp=1, user_ip='1.1.1.1'))
        @param answer: значение КО в формате
            dict(value='new answer', origin_info=dict(timestamp=1, user_ip='1.1.1.1'))
        @param current_answer_info: объект, содержащий информацию о текущем КО.
        @param use_origin_info_from_answer: признак получения информации о времени и IP изменения из объекта answer.
        @return объект, содержащий информацию о новом текущем КО.
        """
        answers = question_to_answers.setdefault(question['value'], OrderedDict())
        origin_info = answer['origin_info'] if use_origin_info_from_answer else question['origin_info']
        interval = {'start': origin_info, 'end': None}
        answer_info = answers.setdefault(answer['value'], dict(intervals=[]))
        question_answer_changed = True
        if answer_info['intervals']:
            last_interval = answer_info['intervals'][-1]
            if last_interval['end']:
                # Этот же ответ (на этот же вопрос) ранее был сменен на другой, теперь устанавливается снова
                answer_info['intervals'].append(interval)
            else:
                # В противном случае, повторно установили тот же ответ на тот же вопрос
                question_answer_changed = False
        else:
            answer_info['intervals'].append(interval)

        # нужно завершить интервал валидности текущего КО
        if current_answer_info and question_answer_changed:
            current_answer_info['intervals'][-1]['end'] = origin_info

        return answer_info

    def post_process_events(self):
        # Отдельно делаем обход, валидирующий собранные данные - иначе код сложно понять
        if not _validate_ts_to_question_answer_sequence(self.ts_to_question_answer):
            return

        question_to_answers = OrderedDict()
        cur_question = None  # КВ до текущего TS на итерации цикла
        cur_answer = None  # КО до текущего TS на итерации цикла
        prev_answer = None  # Предыдущее значение answer в словаре ts_to_question_answer
        prev_question = None  # Предыдущее значение question в словаре ts_to_question_answer
        prev_ts = None
        current_answer_info = None

        # В цикле ниже предполагаем, что данные валидны
        for ts, question_answer in iteritems(self.ts_to_question_answer):
            pair = False
            question = question_answer['question']
            answer = question_answer['answer']
            if prev_ts and abs(prev_ts - ts) > MAX_HINTQ_HINTA_TS_OFFSET:
                # обрабатываем данные с предыдущей итерации, случаи смены только КВ или только КО
                # обязательно до обработки "правильных" случаев, для правильного обновления current_answer_info
                if not prev_question and prev_answer:
                    current_answer_info = self._update_question_answer_intervals(
                        question_to_answers,
                        cur_question,
                        prev_answer,
                        current_answer_info,
                    )
                elif not prev_answer and prev_question:
                    current_answer_info = self._update_question_answer_intervals(
                        question_to_answers,
                        prev_question,
                        cur_answer,
                        current_answer_info,
                        use_origin_info_from_answer=False,
                    )
            if question and answer:
                # правильный случай - установлены КВ и КО в один момент времени
                current_answer_info = self._update_question_answer_intervals(
                    question_to_answers,
                    question,
                    answer,
                    current_answer_info,
                )
            if prev_ts and abs(prev_ts - ts) <= MAX_HINTQ_HINTA_TS_OFFSET:
                # не совсем правильный случай - установлены КВ и КО с небольшим сдвигом по времени
                if prev_question and not question:
                    current_answer_info = self._update_question_answer_intervals(
                        question_to_answers,
                        prev_question,
                        answer,
                        current_answer_info,
                    )
                elif prev_answer and not answer:
                    current_answer_info = self._update_question_answer_intervals(
                        question_to_answers,
                        question,
                        prev_answer,
                        current_answer_info,
                        use_origin_info_from_answer=False,
                    )
                pair = True

            cur_question = question or cur_question
            cur_answer = answer or cur_answer
            prev_ts = ts
            prev_question = question
            prev_answer = answer

        if self.ts_to_question_answer and not pair:
            if not prev_question and prev_answer:
                self._update_question_answer_intervals(
                    question_to_answers,
                    cur_question,
                    prev_answer,
                    current_answer_info,
                )
            if not prev_answer and prev_question:
                self._update_question_answer_intervals(
                    question_to_answers,
                    prev_question,
                    cur_answer,
                    current_answer_info,
                    use_origin_info_from_answer=False,
                )
        return dict(
            question_answer_mapping=question_to_answers,
        )


def serialize_question_answer_mapping(question_answer_mapping):
    """
    Преобразовать отображение КВ в КО в формат, сериализуемый в JSON, с сохранением порядка
    """
    return [
        dict(
            question=question,
            answers=[dict(info, value=answer) for (answer, info) in iteritems(answers)],
        ) for question, answers in iteritems(question_answer_mapping)
    ]


def flatten_question_answer_mapping(serialized_question_answer_mapping):
    """
    На основе сериализованного представления отображения КВ в КО построить список пар КВ/КО упорядоченный
    по интервалам в порядке возрастания timestamp начала интервала
    """
    qa_list = []
    for question_info in serialized_question_answer_mapping:
        for answer_info in question_info['answers']:
            for interval in answer_info['intervals']:
                qa_list.append({
                    'question': question_info['question'],
                    'answer': answer_info['value'],
                    'interval': interval,
                })
    qa_list.sort(key=lambda item: item['interval']['start']['timestamp'])
    return qa_list


def _validate_ts_to_question_answer_sequence(ts_to_question_answer):
    """
    Проверим, что выполняются следующие ожидания:
    1) парные значения КВ/КО задаются либо вместе (на одном TS), либо одно за другим с
    интервалом не более MAX_HINTQ_HINTA_TS_OFFSET
    2) парные значения не могут смешиваться - т.е. не может быть три события подряд
    со сдвигом до MAX_HINTQ_HINTA_TS_OFFSET
    3) в начале всегда задано парное значение КВ/КО
    """
    prev_ts = None
    prev_answer = None
    prev_question = None
    grouped_pairs = OrderedDict()
    for ts, question_answer in iteritems(ts_to_question_answer):
        question = (question_answer['question'] or {}).get('value')
        answer = (question_answer['answer'] or {}).get('value')
        if question and answer:
            grouped_pairs[ts] = (question, answer)
        if prev_ts and abs(prev_ts - ts) <= MAX_HINTQ_HINTA_TS_OFFSET:
            if question and answer or prev_ts in grouped_pairs:
                log.warning('Bad hintq/hinta events: unexpected order')
                return False
            grouped_pairs[prev_ts] = (prev_question or question, prev_answer or answer)
            grouped_pairs[ts] = grouped_pairs[prev_ts]
            flags = tuple(bool(field) for field in (prev_question, prev_answer, question, answer))
            if flags != (True, False, False, True) and flags != (False, True, True, False):
                log.warning('Bad hintq/hinta events: unexpected order')
                return False
        prev_ts = ts
        prev_question = question
        prev_answer = answer
    if ts_to_question_answer:
        # TODO: нужно научиться пропускать ошибки по ненахождению первичных пар hintq/hinta - для старых пользователей
        # первичные данные по КВ/КО могли записываться некорректно, при этом КВ/КО могли сменяться позже, и в текущей
        # реализации мы эту информацию не извлечем.
        first_ts = list(ts_to_question_answer.keys())[0]
        group_ts, (question, answer) = list(grouped_pairs.items())[0] if grouped_pairs else (None, (None, None))
        if not grouped_pairs or group_ts != first_ts or not question or not answer:
            log.warning('Bad hintq/hinta events: cannot find initial question and answer pair')
            return False
    return True
