# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from passport.backend.core.historydb.analyzer.event_handlers.base import EventHandler
from passport.backend.core.historydb.events import (
    ACTION_RESTORE_PASSED_BY_METHOD_PREFIX,
    ACTION_RESTORE_PASSED_BY_METHOD_REGEXP,
    ACTION_RESTORE_SEMI_AUTO_DECISION,
    ACTION_RESTORE_SEMI_AUTO_REQUEST,
    EVENT_ACTION,
    EVENT_INFO_RESTORE_ID,
    EVENT_INFO_RESTORE_REQUEST_SOURCE,
    EVENT_INFO_RESTORE_STATUS,
    EVENT_INFO_SUPPORT_LINK_TYPE,
    RESTORE_STATUS_REJECTED,
)
from six import iteritems


log = logging.getLogger('passport.historydb.analyzer.event_handlers.restore')


class RestorePassedAttemptsHandler(EventHandler):
    """
    Успешные прохождения восстановления (все способы). Работает только с событиями нового рестора.
    """
    events = (
        EVENT_ACTION,
        EVENT_INFO_SUPPORT_LINK_TYPE,
    )

    def __init__(self, *args, **kwargs):
        self.events_by_ts = OrderedDict()
        super(RestorePassedAttemptsHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        if (event['name'] == EVENT_ACTION and
                not event.get('value', '').startswith(ACTION_RESTORE_PASSED_BY_METHOD_PREFIX)):
            return
        attempt = self.events_by_ts.setdefault(event['timestamp'], {})
        if event['name'] == EVENT_ACTION:
            passed_method = ACTION_RESTORE_PASSED_BY_METHOD_REGEXP.match(event['value']).group('method')
            attempt.update(method=passed_method, timestamp=event['timestamp'])
        elif event['name'] == EVENT_INFO_SUPPORT_LINK_TYPE:
            attempt['link_type'] = event['value']

    def post_process_events(self):
        return dict(restore_passed_attempts=[a for a in self.events_by_ts.values() if 'method' in a])


class RestoreSemiAutoAttemptsHandler(EventHandler):
    """
    Попытки восстановления через анкету и их результат.
    В этом коде меньше проверок валидности данных, т.к. писать эти данные в HistoryDB начали недавно.
    """
    events = (
        EVENT_ACTION,
        EVENT_INFO_RESTORE_ID,
        EVENT_INFO_RESTORE_STATUS,
        EVENT_INFO_RESTORE_REQUEST_SOURCE,
    )

    def __init__(self, *args, **kwargs):
        self.events_by_ts = OrderedDict()
        super(RestoreSemiAutoAttemptsHandler, self).__init__(*args, **kwargs)

    def handle_event(self, event):
        expected_actions = (ACTION_RESTORE_SEMI_AUTO_REQUEST, ACTION_RESTORE_SEMI_AUTO_DECISION)
        if event['name'] == EVENT_ACTION and event.get('value') not in expected_actions:
            return
        attempt = self.events_by_ts.setdefault(event['timestamp'], {})
        attempt[event['name']] = event.get('value')
        if event.get('admin'):
            attempt['admin'] = event['admin']

    def post_process_events(self):
        combined_attempts = OrderedDict()
        for ts, attempt in iteritems(self.events_by_ts):
            restore_id = attempt[EVENT_INFO_RESTORE_ID]
            if restore_id not in combined_attempts and attempt[EVENT_ACTION] == ACTION_RESTORE_SEMI_AUTO_REQUEST:
                # Первое событие - запись от анкеты с назначенным первоначальным статусом
                combined_attempts[restore_id] = dict(
                    timestamp=ts,
                    restore_id=restore_id,
                    support_decisions=[],
                    # Писать статус и источник начали не сразу, подставляем разумные умолчания
                    initial_status=attempt.get(EVENT_INFO_RESTORE_STATUS, RESTORE_STATUS_REJECTED),
                    request_source=attempt.get(EVENT_INFO_RESTORE_REQUEST_SOURCE, 'restore'),
                )
            elif restore_id in combined_attempts and attempt[EVENT_ACTION] == ACTION_RESTORE_SEMI_AUTO_DECISION:
                # Последующие события содержат информацию о решениях саппорта
                decision = dict(
                    timestamp=ts,
                    status=attempt[EVENT_INFO_RESTORE_STATUS],
                    admin=attempt['admin'],
                )
                combined_attempts[restore_id]['support_decisions'].append(decision)
            else:
                log.warning(u'Unexpected restore attempt: %s', attempt)

        return dict(restore_semi_auto_attempts=list(combined_attempts.values()))
