# -*- coding: utf-8 -*-
# все, что связано с обработкой результатов ручки /events
from collections import namedtuple
from datetime import datetime
import logging
import time

from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.conf import settings
from passport.backend.core.historydb.analyzer.event_handlers import KWARGS_TO_HANDLERS_EVENTS
from passport.backend.core.historydb.events import (
    EVENT_INFO_FIRSTNAME,
    EVENT_INFO_HINTA,
    EVENT_INFO_HINTQ,
    EVENT_INFO_LASTNAME,
    EVENT_USERINFO_FT,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import datetime_to_unixtime
from six import (
    iteritems,
    itervalues,
)


log = logging.getLogger('passport.historydb.analyzer.events')

# Отображение имен событий в соответствующие поля события userinfo_ft
REGULAR_EVENT_NAME_TO_USERINFO_FT_FIELD = {
    EVENT_INFO_FIRSTNAME: 'firstname',
    EVENT_INFO_LASTNAME: 'lastname',
    EVENT_INFO_HINTQ: 'hintq',
    EVENT_INFO_HINTA: 'hinta',
}

DEFAULT_EVENTS_LIMIT = 5000

DEFAULT_FROM_UNIXTIME = datetime_to_unixtime(datetime(settings.YANDEX_FOUNDATION_YEAR, 1, 1))

EVENTS_INFO_FIELDS = (
    'firstnames',
    'lastnames',
    'birthdays',
    'registration_env',
    'confirmed_phones',
    'emails',
    'confirmed_emails',
    'question_answer_mapping',
    'questions',
    'answers',
    'restore_passed_attempts',
    'restore_semi_auto_attempts',
    'password_change_requests',
    'password_changes',
    'password_hashes',
    'account_enabled_status',
    'account_create_event',
    'account_delete_event',
    'names',
    'app_key_info',
)

EventsInfo = namedtuple('EventsInfo', EVENTS_INFO_FIELDS)


def get_account_registration_unixtime(reg_datetime):
    """PASSP-9876 Не всегда известна дата регистрации пользователя"""
    if reg_datetime is Undefined:
        return int(DEFAULT_FROM_UNIXTIME)

    return int(datetime_to_unixtime(reg_datetime))


class EventsAnalyzer(object):
    def __init__(self, **entities_to_analyze):
        """
        Подготавливаем имена событий и обработчики по набору аргументов (возможные значения
        описаны в KWARGS_TO_HANDLERS_EVENTS).
        """
        event_names_for_request = set()
        event_name_prefixes = set()
        event_to_handler_classes = {}

        for field, (handler_cls, events) in iteritems(KWARGS_TO_HANDLERS_EVENTS):
            is_field_required = entities_to_analyze.pop(field, None)
            if is_field_required:
                event_names_for_request.update(events)
                for event in events:
                    event_to_handler_classes.setdefault(event, set()).add(handler_cls)
                    if event[-1] == '*':
                        event_name_prefixes.add(event[:-1])
        if entities_to_analyze:
            raise ValueError('Got unexpected entities: %s', entities_to_analyze)

        self.event_names_for_request = list(event_names_for_request)
        self.event_name_prefixes = event_name_prefixes
        self.event_to_handler_classes = event_to_handler_classes

    def load_and_analyze_events(self, account=None, uid=None, limit=DEFAULT_EVENTS_LIMIT, **kwargs):
        """
        Загрузить события для аккаунта из HistoryDB, выполнить обработку событий.
        @param kwargs: аргументы для передачи в обработчики событий
        @return объект типа EventsInfo с данными, извлеченными из событий
        """
        if account is None and uid is None:
            raise ValueError('At least UID or account must be specified')
        events = self._load_events(account=account, uid=uid, limit=limit)
        return self._analyze_events(events, **kwargs)

    def _load_events(self, account=None, uid=None, limit=DEFAULT_EVENTS_LIMIT):
        """
        Выполнить загрузку необходимых событий для аккаунта из HistoryDB.
        """
        historydb = get_historydb_api()
        start_ts = 0
        if account is not None:
            start_ts = get_account_registration_unixtime(account.registration_datetime)
            uid = account.uid
        end_ts = int(time.time())
        events = historydb.events(
            uid,
            start_ts,
            end_ts,
            limit=limit,
            name=','.join(sorted(self.event_names_for_request)),
            # Обработчики событий предполагают, что timestamp увеличивается
            ascending=True,
        )
        if len(events) > limit / 2:
            log.warning('Event limit insufficient for UID %s. Increase the limit!', uid)
        return events

    def _analyze_events(self, events, **kwargs):
        """
        Выполнить обработку событий, полученных из HistoryDB.
        @param events: список событий, полученных из HistoryDB
        @return объект типа EventsInfo с данными, извлеченными из событий
        """
        events = _repair_registration_events(events)
        per_event_handlers = self._build_per_event_handlers(events, **kwargs)
        for event in events:
            name = event['name']
            # Получаем обработчики для событий с данным именем
            event_handlers = per_event_handlers.get(name, set())
            for handler in event_handlers:
                handler.handle_event(event)

        results = {}
        for handler in set().union(*itervalues(per_event_handlers)):
            results.update(handler.post_process_events() or {})

        return _construct_events_info(results)

    def _build_per_event_handlers(self, events, **kwargs):
        """
        Подготовить обработчики для событий. Учитываются запрошенные события (в т.ч. префиксы),
        а также полученные события (для учета префиксов).
        """
        per_event_handlers = {}
        # Инициализируем обработчики
        all_handler_classes = set().union(*itervalues(self.event_to_handler_classes))
        handler_instances = {cls: cls(**kwargs) for cls in all_handler_classes}
        # Сопоставим имени события множество обработчиков, ожидающих такие события
        for event_name, handler_classes in iteritems(self.event_to_handler_classes):
            per_event_handlers[event_name] = {handler_instances[cls] for cls in handler_classes}
        # Получим все имена событий, пришедшие в ответе HistoryDBApi. Дополним обработчики событий с учетом
        # запрошенных префиксов имен.
        incoming_event_names = {event['name'] for event in events}
        for event_name in incoming_event_names:
            for name_prefix in self.event_name_prefixes:
                if event_name.startswith(name_prefix):
                    per_event_handlers.setdefault(event_name, set()).update(per_event_handlers[name_prefix + '*'])
        return per_event_handlers


def _construct_events_info(values):
    params = {}
    for field in EVENTS_INFO_FIELDS:
        default = []
        if field in ('emails', 'confirmed_emails', 'registration_env', 'question_answer_mapping'):
            default = {}
        params[field] = values.get(field, default)

    return EventsInfo(**params)


REGISTRATION_TIMESTAMP_OFFSET = 2.0


def _repair_registration_events(events):
    """
    При переливке из таблицы userinfo_ft в HistoryDB возникли ситуации, когда событие
    userinfo_ft записано вместе с событиями об изменении аккаунта. При этом время события userinfo_ft
    округлено до секунд, что создает неточности при анализе событий, так как события об изменении
    аккаунта содержат время с плавающей точкой.
    Кроме того, в пределах регистрации бывает ситуация, что несколько событий имеют различный плавающий timestamp:
     1343053554         userinfo_ft
     1343053554.0141    info.password_quality
     1343053554.04411   sid.add
     1343053554.04411   info.password
     ...
    Для решения проблемы пытаемся привести timestamp регистрации в событиях к одному значению.
    Убираем дубликаты данных - имя, фамилия, КВ, КО, но оставляем их в событии userinfo_ft - многое завязано
    на это событие, в т.ч. определение факта регистрации аккаунта.
    """
    if not events or len(events) < 2:
        return events
    first_timestamp = events[0]['timestamp']
    userinfo_index = None
    for index, event in enumerate(events):
        # ищем событие userinfo_ft в первый момент времени
        # в большинстве случаев это первое событие, но могут быть исключения, когда округленное время
        # совпало с неокругленным
        if event['timestamp'] > first_timestamp:
            break
        if event['name'] == EVENT_USERINFO_FT:
            userinfo_index = index
            break
    if userinfo_index is None:
        # нет события userinfo_ft, не наш случай
        return events
    userinfo_ft = events.pop(userinfo_index)
    visited_event_names = set()
    repaired_events = []
    first_usual_event_index = None
    for index, event in enumerate(events):
        timestamp = event['timestamp']
        name = event['name']
        if timestamp >= first_timestamp + REGISTRATION_TIMESTAMP_OFFSET or name in visited_event_names:
            # событие не относится к регистрации, либо событие уже встречалось
            first_usual_event_index = index
            break
        visited_event_names.add(name)

        if name in REGULAR_EVENT_NAME_TO_USERINFO_FT_FIELD:
            ft_field = REGULAR_EVENT_NAME_TO_USERINFO_FT_FIELD[name]
            ft_value = userinfo_ft.get(ft_field)
            if name != EVENT_INFO_HINTQ and ft_value != event.get('value'):
                # несовпадение КВ - это нормально, т.к. в событии info.hintq не пишется текст вопроса, только ID
                log.warning('userinfo_ft field %s and regular event value mismatch', ft_field)
            # не записываем это событие в результат, чтобы не нарушать консистентность
            continue
        event['timestamp'] = first_timestamp
        repaired_events.append(event)

    if repaired_events:
        log.debug('Repaired %d registration events', len(repaired_events))
    unchanged_events = events[first_usual_event_index:] if first_usual_event_index is not None else []
    return [userinfo_ft] + repaired_events + unchanged_events


__all__ = (
    'EventsAnalyzer',
    'EventsInfo',
)
