# -*- coding: utf-8 -*-

from passport.backend.core.historydb.crypto import encrypt as encrypt_value
from passport.backend.core.historydb.exceptions import RequiredAttributeError
from passport.backend.core.logging_utils.helpers import escape_text_for_log
from passport.backend.utils.string import (
    always_str,
    smart_bytes,
)
from passport.backend.utils.time import datetime_to_string
import pytz
import six


class EntryConverter(object):
    """
    Генерирует строчки для логирования в history db
    http://wiki.yandex-team.ru/passport/history/logs-format
    """
    LOCAL_TZ = pytz.timezone('Europe/Moscow')

    fields = []  # поля для обработки
    datetime_fields = {'time'}  # поля, значения которых являются временем
    entry_required_fields = set()
    escape_fields = set()
    quote_fields = set()
    nullify_fields = set()
    hex_fields = {'host_id'}  # поля, значения которых должны быть представлены в виде hex строки

    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%f%z'
    NULL = '-'
    FIELD_SEPARATOR = ' '

    def get_field(self, entry, field_name):
        try:
            return getattr(entry, field_name)
        except (KeyError, AttributeError):
            return None

    def convert(self, entry):
        """
        Преобразование entry в строку для логирования
        :type entry: LogEntry
        """
        self.check_required_fields(entry)

        msg_parts = []
        sensitive_fields = entry.sensitive_fields
        for field in self.fields:
            value = self.get_field(entry, field)
            msg_part = self.convert_field(field, value, encrypt=field in sensitive_fields)
            msg_parts.append(smart_bytes(msg_part))

        result = self.FIELD_SEPARATOR.encode('utf8').join(msg_parts)
        if six.PY3:
            result = result.decode('utf8')
        return result

    def convert_field(self, name, value, encrypt=False):
        """
        Преобразование поля в строку для логирования
        :param name: имя поля
        :param value: значение поля
        :param encrypt: требуется ли зашифровать значение
        """
        if name in self.datetime_fields:
            value = self.LOCAL_TZ.localize(value)
            # FIXME: Убрать [:-2] когда historydb-loader будет готов к парсингу +HHMM
            value = datetime_to_string(value, format=self.DATETIME_FORMAT)[:-2]
        elif value is None:
            return self.nullify(value)
        elif encrypt and value:
            return encrypt_value(value)

        if name in self.hex_fields:
            value = self.hexify(value)
        if name in self.escape_fields:
            value = self.escape(value)
        if name in self.quote_fields:
            value = self.quote(value)
        if name in self.nullify_fields:
            value = self.nullify(value)
        return value

    def escape(self, value):
        old = ['\n', '\r']
        new = ['\\n', '\\r']
        return self.replace(old, new, value)

    def quote(self, value):
        if value == '-':
            return '`%s`' % value

        old = ['`']
        new = ['``']

        if ' ' in value or '`' in value:
            value = '`%s`' % self.replace(old, new, value)

        return value

    def nullify(self, value):
        if value == '' or value is None:
            return self.NULL
        return value

    def hexify(self, value):
        return ('%x' % value).upper()

    def replace(self, olds, news, value):
        """
        Замена подстрок в строке
        :param olds: итерируемый список подстрок, которые будут заменены
        :param news: итерируемый список подстрок,на которые будут заменены olds
        :param value: строка в которой происходит замена
        """
        for old, new in zip(olds, news):
            value = value.replace(old, new)
        return value

    def check_required_fields(self, entry):
        for required_field in self.entry_required_fields:
            if not hasattr(entry, required_field):
                raise RequiredAttributeError(required_field)


class AuthEntryConverter(EntryConverter):
    """authnew.log нового формата"""

    fields = [
        'version',
        'time',
        'host_id',
        'client_name',
        'uid',
        'login',
        'sid',
        'type',
        'status',
        'comment',
        'ip_from',
        'ip_prox',
        'yandexuid',
        'referer',
        'retpath',
        'useragent',
    ]

    escape_fields = {
        'login',
        'referer',
        'retpath',
        'useragent',
        'yandexuid',
    }
    quote_fields = {
        'login',
        'referer',
        'retpath',
        'useragent',
        'yandexuid',
    }
    nullify_fields = {
        'uid',
        'login',
        'sid',
        'ip_from',
        'ip_prox',
        'yandexuid',
        'referer',
        'retpath',
        'useragent',
        'comment',
    }
    entry_required_fields = {
        'time',
        'host_id',
        'client_name',
        'uid',
        'type',
        'status',
        'ip_from',
    }


class EventEntryConverter(EntryConverter):
    """event.log нового формата"""

    fields = [
        'version',
        'time',
        'host_id',
        'client_name',
        'uid',
        'name',
        'value',
        'ip_from',
        'ip_prox',
        'yandexuid',
        'admin',
        'comment',
    ]
    escape_fields = {
        'value',
        'yandexuid',
        'admin',
        'comment',
    }
    quote_fields = {
        'value',
        'yandexuid',
        'admin',
        'comment',
    }
    nullify_fields = {
        'value',
        'ip_from',
        'ip_prox',
        'yandexuid',
        'admin',
        'comment',
    }
    entry_required_fields = {
        'time',
        'host_id',
        'client_name',
        'uid',
        'name',
        'value',
        'ip_from',
    }


class RestoreEntryConverter(EntryConverter):
    """restore.log"""
    fields = [
        'version',
        'action',
        'time',
        'uid',
        'restore_id',
        'data_json',
    ]
    escape_fields = {'data_json'}
    quote_fields = {'data_json'}
    nullify_fields = {'data_json'}
    entry_required_fields = {
        'action',
        'time',
        'uid',
        'restore_id',
        'data_json',
    }


class AuthChallengeEntryConverter(EntryConverter):
    """auth_challenge.log"""
    fields = [
        'version',
        'time',
        'action',
        'uid',
        'env_profile_id',
        'env_profile_pb2_base64',
        'env_json',
        'comment',
    ]
    escape_fields = {
        'env_json',
        'comment',
    }
    quote_fields = {
        'env_json',
        'comment',
    }
    nullify_fields = {
        'env_profile_pb_base64',
        'env_json',
        'comment',
    }
    entry_required_fields = {
        'action',
        'time',
        'uid',
        'env_profile_id',
        'env_profile_pb2_base64',
        'env_json',
    }


class EntryTskvConverter(object):
    """
    Генерирует строчки для логирования в history db в формате tskv
    """
    LOCAL_TZ = pytz.timezone('Europe/Moscow')

    fields = []  # поля для обработки
    datetime_fields = {'time'}  # поля, значения которых являются временем
    entry_required_fields = set()
    escape_fields = set()
    hex_fields = {'host_id'}  # поля, значения которых должны быть представлены в виде hex строки
    DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%f%z'
    KV_FORMAT = '{key}={value}'
    FIELD_SEPARATOR = '\t'

    def get_field(self, entry, field_name):
        try:
            return entry.data[field_name]
        except (KeyError, AttributeError):
            return None

    def convert(self, entry):
        """
        Преобразование entry в строку для логирования
        :type entry: LogEntry
        """
        # TODO рассмотреть возможноть интеграции
        #      https://a.yandex-team.ru/arcadia/passport/backend/core/logging_utils/loggers/tskv.py?rev=r9115534#L68
        self.check_required_fields(entry)
        msg_parts = []
        sensitive_fields = entry.sensitive_fields
        # Сначало маппим KV эвентов так как в TSKV это поля с динамическими значениями
        for field, value in entry.events.items():
            msg_part = self.convert_field(field, value='' if value is None else value, encrypt=field in sensitive_fields)
            if msg_part is None:
                # Значение должно быть иначе что-то пошло не так
                raise ValueError('Event name should have value')
            msg_parts.append(always_str(msg_part))
        # Теперь маппим дефолтные поля
        for field in self.fields:
            value = self.get_field(entry, field)
            msg_part = self.convert_field(field, value, encrypt=field in sensitive_fields)
            if msg_part is None:
                continue
            msg_parts.append(always_str(msg_part))

        return self.FIELD_SEPARATOR.join(msg_parts)

    def convert_field(self, name, value, encrypt=False):
        """
        Преобразование поля в строку для логирования
        :param name: имя поля
        :param value: значение поля
        :param encrypt: требуется ли зашифровать значение
        """
        if name in self.datetime_fields:
            value = self.LOCAL_TZ.localize(value)
            value = datetime_to_string(value, format=self.DATETIME_FORMAT)[:-2]
        elif value is None:
            return None
        elif encrypt and value:
            return self.KV_FORMAT.format(name, encrypt_value(value))

        if name in self.hex_fields:
            value = self.hexify(value)
        if name in self.escape_fields:
            value = escape_text_for_log(value)

        result = self.KV_FORMAT.format(key=name, value=always_str(value))
        return result

    def hexify(self, value):
        return ('%x' % value).upper()

    def replace(self, olds, news, value):
        """
        Замена подстрок в строке
        :param olds: итерируемый список подстрок, которые будут заменены
        :param news: итерируемый список подстрок,на которые будут заменены olds
        :param value: строка в которой происходит замена
        """
        for old, new in zip(olds, news):
            value = value.replace(old, new)
        return value

    def check_required_fields(self, entry):
        for required_field in self.entry_required_fields:
            if required_field not in entry.data:
                raise RequiredAttributeError(required_field)


class EventEntryTskvConverter(EntryTskvConverter):
    """event.log нового формата"""

    fields = [
        'version',
        'time',
        'host_id',
        'client_name',
        'uid',
        'ip_from',
        'ip_prox',
        'yandexuid',
        'user_agent',
        'admin',
        'comment',
    ]
    escape_fields = {
        'value',
        'yandexuid',
        'admin',
        'comment',
    }

    entry_required_fields = {
        'time',
        'host_id',
        'client_name',
        'uid',
    }
