# -*- coding: utf-8 -*-
from datetime import datetime
import json
import logging

from passport.backend.core.conf import settings
from passport.backend.core.historydb.converter import (
    AuthEntryConverter,
    EventEntryConverter,
    EventEntryTskvConverter,
)
from passport.backend.core.types.ip.ip import IP
from six import (
    iteritems,
    string_types,
)


log = logging.getLogger('passport.historydb.entry')

EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION = {
    1: set(),
    2: {'info.password', 'info.hinta'},
}


class LogEntry(object):
    def __init__(self, uid=None, user_ip=None, proxy_ip=None, comment=None, host_id=None,
                 client_name=None, yandexuid=None, time=None):
        self.version = 1
        self.uid = uid
        self.user_ip = user_ip  # ip_from
        self.proxy_ip = proxy_ip  # ip_prox
        self.comment = comment
        self.host_id = host_id  # уникальный шестнадцатиричный идентификатор сервера, на котором сделана запись
        self.client_name = client_name  # имя сервиса, сделавшего запись (например, passport)
        self.yandexuid = yandexuid
        self.time = time or datetime.now()  # datetime а не timestamp

    @property
    def ip_from(self):

        # TODO: Возможно стоит хранить сразу нормализованное значение в модели и не допускать свойств
        return IP(self.user_ip).normalized

    @property
    def ip_prox(self):
        """
        deprecated
        proof: cat /var/log/yandex/passport-api/historydb/event.log | awk '{print $(NF-3)}' | grep -v - | wc -l
        """
        return IP(self.proxy_ip).normalized

    @property
    def sensitive_fields(self):
        return set()

    def __unicode__(self):
        return str(self).decode('utf-8')

    def __str__(self):
        raise NotImplementedError()


class AuthEntry(LogEntry):
    """
     Обязательные поля: version time host-id client-name uid type status ip-from
    """
    def __init__(self, login=None, sid=None, type=None, status=None, retpath=None, useragent=None, comment=None, **kwargs):
        self.login = login
        self.sid = sid
        self.type = type  # тип авторизации - WEB, TOKEN, OAUTH
        self.status = status  # FAILED, SESSION_KILL, SESSION_CREATE, SESSION_UPDATE
        self.referer = None  # TODO: удалить в следующей версии формата
        self.retpath = retpath
        self.useragent = useragent
        super(AuthEntry, self).__init__(comment=self.format_comment_dict(comment), **kwargs)

    @staticmethod
    def format_comment_dict(comment):
        if comment is None or isinstance(comment, string_types):
            return comment

        return ';'.join('%s=%s' % (k, v) for k, v in sorted(iteritems(comment)))

    def __str__(self):
        return AuthEntryConverter().convert(self)


class LogTskvEntry(object):
    def __init__(self, events, **kwargs):
        self.data = kwargs
        # Что бы не доставать потом в sensitive_fields парсингом
        self.events = events
        self.data['version'] = 1
        self.data.setdefault('time', datetime.now())  # datetime а не timestamp

    def __unicode__(self):
        return str(self).decode('utf-8')

    def __str__(self):
        raise NotImplementedError()


class EventTskvEntry(LogTskvEntry):
    def __init__(self,  **kwargs):
        super(EventTskvEntry, self).__init__(**kwargs)
        self.data['version'] = 2 if settings.HISTORYDB_LOG_ENCRYPTION_ENABLED else 1

    @property
    def sensitive_fields(self):
        fields = set()
        for name, value in self.events.items():
            if name in EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION[self.data['version']]:
                fields.add(value)
        return fields

    def __str__(self):
        return EventEntryTskvConverter().convert(self)


class EventEntry(LogEntry):
    """
     Обязательные поля: version time host-id client-name uid name value ip-from
    """
    def __init__(self, admin=None, name=None, value=None, **kwargs):
        super(EventEntry, self).__init__(**kwargs)
        self.version = 2 if settings.HISTORYDB_LOG_ENCRYPTION_ENABLED else 1
        self.admin = admin  # логин администратора
        self.name = name  # имя события
        self.value = value  # значение события

    @property
    def sensitive_fields(self):
        if self.name in EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION[self.version]:
            return {'value'}
        return set()

    def __str__(self):
        return EventEntryConverter().convert(self)


class LoyaltyEntry(LogEntry):
    def __init__(self, timestamp=None, user_id_type=None, user_id=None, user_ip=None,
                 domain=None, name=None, meta=None):
        self.timestamp = timestamp
        self.user_id_type = user_id_type
        self.user_id = user_id
        self.domain = domain
        self.name = name
        self.meta = meta
        super(LoyaltyEntry, self).__init__(user_ip=user_ip)

    @property
    def meta_json(self):
        if self.meta is not None:
            return json.dumps(self.meta)

    @property
    def timestamp_str(self):
        return '%.6f' % self.timestamp


class RestoreEntry(LogEntry):
    def __init__(self, action=None, uid=None, restore_id=None, data=None):
        super(RestoreEntry, self).__init__(uid=uid)
        self.version = 2 if settings.HISTORYDB_LOG_ENCRYPTION_ENABLED else 1
        self.action = action
        self.restore_id = restore_id
        self.data = data

    @property
    def data_json(self):
        if self.data is not None:
            return json.dumps(self.data)

    @property
    def sensitive_fields(self):
        if self.version == 2:
            return {'data_json'}
        return set()


class AuthChallengeEntry(LogEntry):
    def __init__(self, action, uid, env_profile_id, env, comment):
        self.action = action
        self.env_profile_id = env_profile_id
        self.env = env
        super(AuthChallengeEntry, self).__init__(uid=uid, comment=comment)

    @property
    def env_profile_pb2_base64(self):
        # Поле не используется
        return '-'

    @property
    def env_json(self):
        try:
            return json.dumps(self.env)
        except:
            log.warning('Failed to dump auth challenge env: %s', repr(self.env))
            raise
