# -*- coding: utf-8 -*-
import logging

from cookiemy import Cookiemy
from langdetect import Langdetect
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import WrongHostError
from passport.backend.core.geobase import get_geobase
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.utils.string import smart_bytes
import six


log = logging.getLogger('passport.core.language_detect')


@lazy_loadable()
class LanguageDetect(object):
    def __init__(self):
        self.lookup = Langdetect(smart_bytes(settings.LANGDETECT_DATA_FILE))

    def __getattr__(self, item):
        return getattr(self.lookup, item)


def get_language_detect():
    return LazyLoader.get_instance('LanguageDetect')


def get_language_number(language):
    language_detect = get_language_detect()
    return language_detect.language2cookie(smart_bytes(language))


def get_preferred_language(selected_language):
    language_request = {
        'filter': smart_bytes(','.join(settings.DISPLAY_LANGUAGES)),
        'default': smart_bytes(settings.ALL_SUPPORTED_LANGUAGES['default']),
        'pass-language': smart_bytes(selected_language or ''),
    }
    return get_language_detect().find(language_request)[0][0].decode()


COOKIEMY_LANGUAGE_BLOCK = 39


def _select_host(host):
    host = host.lower()
    for i in settings.DETECT_LANGUAGES.keys():
        if host.endswith(i):
            return i
    raise WrongHostError("Can't found an appropriate domain for %s in config." % host)


class LanguageData(object):
    def __init__(self, ip='', cookiemy='', yandex_gid='', host='', accept_language='',
                 language_detect=None, geobase=None):
        self.ip = ip
        if isinstance(cookiemy, six.text_type):
            self.cookiemy = cookiemy.encode('ascii')
        else:
            self.cookiemy = cookiemy

        self.default_language_settings = {'all': ['ru'], 'default': 'ru'}
        if host:
            try:
                host = _select_host(host)
            except WrongHostError as ex:
                log.debug('Wrong host: %s', ex)
                self.default_language_settings = settings.ALL_SUPPORTED_LANGUAGES

        self.host = host
        self.yandex_gid = yandex_gid
        self.accept_language = accept_language
        self.language_detect = language_detect or get_language_detect()
        self.geobase = geobase or get_geobase()

    def get_lang_from_cookiemy(self):
        try:
            block = self.cookiemy_parser.find(COOKIEMY_LANGUAGE_BLOCK)
            return str(block[1]) if block and len(block) > 1 else '0'
        except RuntimeError:  # На кривую куку кидается RuntimeError
            return '0'

    def create_language_detect_request(self, language_settings, domain=None):
        request = {
            'geo': smart_bytes(self.current_location),
            'cookie': smart_bytes(self.get_lang_from_cookiemy()),
            'filter': smart_bytes(','.join(language_settings.get('all'))),
            'default': smart_bytes(language_settings.get('default')),
            'language': smart_bytes(self.accept_language),
        }
        if domain:
            request['domain'] = smart_bytes(domain)
        return request

    @property
    def current_location(self):
        if self.yandex_gid is not None and self.yandex_gid.isdigit():
            return self.yandex_gid

        try:
            return ','.join(str(x['id']) for x in self.geobase.regions(self.ip))
        except (ValueError, RuntimeError, TypeError):
            return ''

    @property
    def cookiemy_parser(self):
        parser = Cookiemy()
        parser.parse(self.cookiemy)
        return parser

    @property
    def language(self):
        if self.host in settings.DETECT_LANGUAGES:
            _language = self.language_detect.find(self.create_language_detect_request(
                settings.DETECT_LANGUAGES[self.host],
                domain=self.host,
            ))  # [('ru', 'Ru')]
        else:
            _language = self.language_detect.find_without_domain(self.create_language_detect_request(
                self.default_language_settings,
            ))
        return _language[0][0].decode('utf8')


class MobileLanguageData(LanguageData):
    def __init__(self, ip='', accept_language='', default_fallback='en'):
        super(MobileLanguageData, self).__init__(ip=ip, accept_language=accept_language)
        self.default_language_settings = {
            'all': settings.DISPLAY_LANGUAGES,
            'default': default_fallback if default_fallback is not None else settings.ALL_SUPPORTED_LANGUAGES['default'],
        }

    @property
    def language(self):
        if self.accept_language:
            if self.accept_language in settings.MOBILE_FALLBACKS:
                return settings.MOBILE_FALLBACKS[self.accept_language]
            else:
                if self.accept_language in self.default_language_settings['all']:
                    return self.accept_language
                else:
                    return self.default_language_settings['default']
        else:
            return super(MobileLanguageData, self).language


class OctopusLanguageData(LanguageData):
    def __init__(self, ip, display_language):
        super(OctopusLanguageData, self).__init__(ip=ip, accept_language=display_language)
        self.default_language_settings = {
            'all': settings.OCTOPUS_SUPPORTED_LANGUAGES,
            'default': settings.ALL_SUPPORTED_LANGUAGES['default'],
        }
