# -*- coding: utf-8 -*-
import copy
import logging

from passport.backend.core.exceptions import BaseCoreError
import six


log = logging.getLogger('passport.lazy_loader')


class Instance(object):
    def __init__(self, name_, type_, *args, **kwargs):
        self._instance = None
        self._name = name_
        self._type = type_
        self._args = args
        self._kwargs = kwargs

    def get_or_create(self, *args, **kwargs):
        if self._instance is None:
            self.force_create(*args, **kwargs)
        return self._instance

    def force_create(self, *args, **kwargs):
        log.info(u'Start initializing %s', self._name)
        try:
            if kwargs:
                for kwargs_name in kwargs:
                    self._kwargs[kwargs_name] = kwargs[kwargs_name]
            if args:
                self._args = args + self._args[len(args):len(self._args)]
            self._instance = self._type(*self._args, **self._kwargs)
        except Exception as e:
            log.info(u'Stop initializing %s, status ERROR', self._name)
            log.error(u'Failed with error %s', six.text_type(e))
            raise
        log.info(u'Stop initializing %s, status OK', self._name)

    def flush(self):
        self._instance = None


class InstanceNotRegisteredError(BaseCoreError):
    """Перед вызовом get_instance необходимо зарегистрировать instance_type"""


class LazyLoader(object):
    _instances = {}

    @staticmethod
    def register(instance_name, instance_type, *args, **kwargs):
        LazyLoader._instances[instance_name] = Instance(instance_name, instance_type, *args, **kwargs)

    @staticmethod
    def get_instance(instance_name, *args, **kwargs):
        if instance_name not in LazyLoader._instances:
            raise InstanceNotRegisteredError(instance_name)
        if kwargs or args:
            kwargs_instance_name = '{a}{b}{c}'.format(
                a=instance_name,
                b=('_' + str(args) if args else ''),
                c=('_' + str(sorted(kwargs.items())) if kwargs else ''),
            )
            if kwargs_instance_name not in LazyLoader._instances:
                LazyLoader._instances[kwargs_instance_name] = copy.deepcopy(LazyLoader._instances[instance_name])
                LazyLoader._instances[kwargs_instance_name].flush()
            return LazyLoader._instances[kwargs_instance_name].get_or_create(*args, **kwargs)

        return LazyLoader._instances[instance_name].get_or_create()

    @staticmethod
    def flush(instance_name=None):
        if instance_name:
            if instance_name in LazyLoader._instances:
                LazyLoader._instances[instance_name].flush()
            else:
                raise InstanceNotRegisteredError()
        else:
            for name in LazyLoader._instances:
                if name not in ('Geobase', 'ASLookup', 'IPReg'):
                    # Файлы геобазы очень большие, перезагрузка стоят дорого
                    LazyLoader._instances[name].flush()

    @staticmethod
    def drop(instance_name):
        if instance_name in LazyLoader._instances:
            LazyLoader._instances.pop(instance_name)
        else:
            raise InstanceNotRegisteredError()


def lazy_loadable(name=None, *args, **kwargs):
    """Декоратор. Позволяет лениво создавать синглтоны декорированного класса с помощью LazyLoader"""

    def decorator(cls):
        LazyLoader.register(name or cls.__name__, cls, *args, **kwargs)
        return cls

    return decorator


__all__ = (
    'LazyLoader',
    'InstanceNotRegisteredError',
    'lazy_loadable',
)
