# -*- coding: utf-8 -*-
import logging
from socket import gethostname
import uuid

from passport.backend.core.conf import settings
from passport.backend.core.logbroker.logbroker_base import format_protobuf_safe
from passport.backend.core.logbroker.logbroker_sdk_client import LogbrokerProducer
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.protobuf.logbroker_passport_protocol.logbroker_passport_protocol_pb2 import (
    PassportProtocolMessage,
)
from passport.backend.utils.importlib import import_object
from passport.backend.utils.string import smart_bytes
from six import with_metaclass


log = logging.getLogger('passport.logbroker')


class NamedSingletonMeta(type):
    _instances = {}

    def __call__(cls, name, *args, **kwargs):
        key = (cls, name)
        if key not in cls._instances:
            obj = super(NamedSingletonMeta, cls).__call__(name, *args, **kwargs)
            cls._instances[key] = obj
            log.debug('Created client instance {}: {}. Instances: {}'.format(
                key, id(obj), {k: id(v) for k, v in cls._instances.items()},
            ))
        return cls._instances[key]


class LogbrokerWriterProto(with_metaclass(NamedSingletonMeta, object)):
    PROTOCOL_VERSION = (1, 0)

    def __init__(self, name, override_host=None, partition_group=None):
        """
        Писатель в лоргброкер.
        Работает как синглтон для каждого name.
        Следует использовать сабкласс из конкретного сервиса, например api:
        api/common/logbroker.py:ApiLogbrokerWriterProto
        там добавляются ценные данные в заголовок.

        Настройки (словарь settings.LOGBROKER_WRITERS[name]):
        * host - хост эндпойнта, str
        * port - порт эндпойнта, int
        * topic - logbroker-топик, str
        * message_class - путь к классу отправляемых сообщений
          (protobuf message), str
        * source_id_prefix - префикс для source id, к нему добавляется uuid4
        * credentials_config - словарь с настройками аутентификации:
            для OAuth (в тестовых целях):
                {'credentials_type': 'oauth_token'}
                требует установки переменной окружения LB_TOKEN

            для TVM:
                {
                    'credentials_type': 'tvm',
                    'tvm_alias': tvm_alias,
                }

        Опциональные настройки:
        * graphite_log - писать в graphite log, bool, default=True
        * connect_timeout - таймаут соединения с API (сек), float, default=0.3
        * write_timeout - таймаут записи сообщений (сек), float, default=0.3
        * extra_fields - словарь значений мета-полей сообщения, Dict[str, str]
        * ca_root_file - путь к файлу с корневыми сертификатами.
                         Если задан - используется TLS

        :param name: имя из настроек LOGBROKER_WRITERS
        :param override_host: явное указание хоста
        :param partition_group: явное указание группы партиций

        :raises KeyError, ValueError: при ошибке в настройках
        """
        try:
            self.conf = settings.LOGBROKER_WRITERS[name]
        except KeyError:
            raise ValueError('No logbroker "{}" in config'.format(name))
        self.message_class = import_object(self.conf['message_class'])
        extra_fields = self.conf.get('extra_fields', {})
        extra_fields.update(
            {
                'message_class': self.message_class.__name__,
                'passport_protocol_v': self._pack_version(),
                'server': gethostname(),
            },
        )
        ca_cert = None
        if 'ca_cert_file' in self.conf:
            with open(self.conf['ca_cert_file']) as f:
                ca_cert = f.read().encode()

        log.debug('Creating logbroker with config {}{}'.format(
            self.conf,
            '' if ca_cert is None else ' (TLS)',
        ))

        self._producer = LogbrokerProducer(
            host=override_host or self.conf['host'],
            port=self.conf['port'],
            topic=self.conf['topic'],
            source_id=smart_bytes(self._source_id()),
            connect_timeout=self.conf.get('connect_timeout', 0.3),
            write_timeout=self.conf.get('write_timeout', 0.3),
            credentials_config=self.conf['credentials_config'],
            extra_fields=extra_fields,
            graphite_logger=GraphiteLogger(service='logbroker'),
            partition_group=partition_group or self.conf.get('partition_group'),
            ca_cert=ca_cert,
        )

    def _source_id(self):
        postfix = str(uuid.uuid4()).split('-')[-1]
        return '{}-{}'.format(self.conf['source_id_prefix'], postfix)

    def _update_header(self, header):
        pass

    def _pack_version(self):
        return '.'.join(str(s) for s in self.PROTOCOL_VERSION)

    def _send_payload(self, payload):
        header = PassportProtocolMessage.Header()
        self._update_header(header)
        message = PassportProtocolMessage(
            message_payload=payload,
            header=header,
        )
        self._producer.write(message.SerializeToString())

    def warm_up(self):
        """
        Прогрев соединения

        :raises: TimeoutError(TransportError) - при таймауте соединения
        :raises: TransportError - при сетевой ошибке
        :raises: ProtocolError - при ошибке на уровне logbroker-протокола
        """
        self._producer.start()

    def send(self, payload):
        """
        Отправить сообщение в Logbroker

        :param payload: protobuf-объект сообщения

        :raises ValueError: если класс payload не соответствует message_class
            из настроек
        :raises TimeoutError(TransportError): при сетевых таймаутах
        :raises TransportError: при сетевых ошибках
        :raises ProtocolError: при ошибках в протоколе взаимодействия с LB
        """
        self._producer.start()
        payload_info = trim_message(format_protobuf_safe(payload))
        log.debug(
            u'Sending data to logbroker {}: {}'.format(
                self._producer.topic, payload_info,
            ),
        )
        if not isinstance(payload, self.message_class):
            # Эта проверка избыточна, но благодаря ней ошибка будет на стороне,
            # на которой она возникла, а не на принимающей стороне
            message = 'payload for {} must be an instance of {} (got {!r})'.format(
                self.__class__.__name__,
                self.message_class.__name__,
                payload,
            )
            log.error(message)
            raise ValueError(message)
        self._send_payload(payload.SerializeToString())


def get_logbroker_writer(name, override_host=None, partition_group=None):
    """
    Получить экземпляр logbroker writer

    :param name: имя writer
    :param override_host: явное указание хоста
    :param partition_group: явное указание группы партиций
    :type name: str
    :return: экземпляр writer
    :rtype: LogbrokerWriterProto
    :raises KeyError, ValueError: при ошибке в настройках
    """
    return LogbrokerWriterProto(
        name, override_host=override_host, partition_group=partition_group,
    )
